; -------------------------------------------------------------------------------------------------------------- ;
; Beach Head - LOCODE.prg
; -------------------------------------------------------------------------------------------------------------- ;
; Memory Map
; -------------------------------------------------------------------------------------------------------------- ;
; $0000 - $00ff:  Zero Page Values
; $0100 - $01ff:  Processor stack
; $0200 - $02ff:  Work Values and C64 system
; $0300 - $03ff:  Work Values and C64 system
; $0400 - $07ff:  Screen: Status
; $0550 - $07a8:  Scores Top Ten
; $0800 - $1dff:  Code: Basic
; $1e00 - $4003:  Code: LOcode  -> $3504=len $1704
; $3c00 - $3fff:  Screen: Save
; $3f00 - $5e03:  Data: Sprites load - will be copied over to $e000-$fdff
; $4000 - $47ff:  Data: Char sets
; $4800 - $4bff:  Screen: 1st
; $4c00 - $4fff:  Screen: 2nd - for scrolling tanks on the beach
; $5000 - $6577:  Data: Sprites game
; $6850 - $c002:  Code: HIcode  -> $bf4f=len $56ff
; $c003 - $cfff:  <unused>
; $e000 - $fdff:  Data: Sprites raw
; $fe00 - $feff:  <unused>
; $ff00 - $ffff:  Kernel vectors
; -------------------------------------------------------------------------------------------------------------- ;
Seg_MAP                 include asm\data\Sprites_map.asm    ; 
Seg_CAVE                include asm\data\Sprites_cave.asm   ; 
Seg_PLANE               include asm\data\Sprites_plane.asm  ; 
Seg_SHIP                include asm\data\Sprites_ship.asm   ; 
Seg_BEACH               include asm\data\Sprites_beach.asm  ; 
Seg_HILL                include asm\data\Sprites_hill.asm   ; 

Seg_SPRITES             seg.u   SPRITES                     ; 
                        include asm\Sprites.asm             ; SP_SpriteDatPtrLo/Hi
; -------------------------------------------------------------------------------------------------------------- ;
                        seg LOCODE
; -------------------------------------------------------------------------------------------------------------- ;
                        * equ $1e00                 ; Beach Head - LOCODE.prg
; -------------------------------------------------------------------------------------------------------------- ;
; compiler settings
; -------------------------------------------------------------------------------------------------------------- ;
                        incdir  ..\inc              ; C64 System Includes

C64CIA1                 include cia1.asm            ; Complex Interface Adapter (CIA) #1 Registers  $DC00-$DC0F
C64CIA2                 include cia2.asm            ; Complex Interface Adapter (CIA) #2 Registers  $DD00-$DD0F
C64SID                  include sid.asm             ; Sound Interface Device (SID) Registers        $D400-$D41C
C64VicII                include vic.asm             ; Video Interface Chip (VIC-II) Registers       $D000-$D02E
C64Kernel               include kernel.asm          ; Kernel Vectors
C64Colors               include color.asm           ; Colour RAM Address / Colours
C64Memory               include mem.asm             ; Memory Layout

Game                    include inc\BH_Game.asm     ; Game Variables
ZeroPage                include inc\BH_Zpg.asm      ; Zero Page Addresses
; -------------------------------------------------------------------------------------------------------------- ;
; GameStart             Does    : Init all - MainLoop - Reset all - Return to BASIC
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GameStart               subroutine                  ; .hbu001. - BASIC 210
                        jsr GameInit                ; 
                        
                        ldx #$00                    ; 
.CopyKernalToRam        lda Kernel_Vectors,x        ; 
                        sta Kernel_Vectors,x        ; 
                        dex                         ; 
                        bne .CopyKernalToRam        ; 
                        
                        jsr C64IrqScrollInit        ; 
                        
                        lda #B_Koff                 ; basic/kernal off
                        sta R6510                   ; 
                        
                        jsr PlayerInit              ; 
                        
.CallMainLoop           jsr Chapter_BayAreaMap      ; with main loop split into chapter loops
                        
                        lda #BIKon                  ; basic/kernal on
                        sta R6510                   ; 
                        
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        jsr C64Reset                ; 
                        
                        lda #$00                    ; 
                        sta CIACRB                  ; CIA1($DC0F) Control Register B
                        
GameStartX              rts                         ; return to BASIC
; -------------------------------------------------------------------------------------------------------------- ;
; C64Reset              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
C64Reset                subroutine                  ; 
                        sei                         ; 
                        lda #$00                    ; 
                        sta IRQMASK                 ; VIC($D01A) IRQ Mask Register
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #VIC_MCM_Off            ; ###.#### - multi color mode
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda #[VIC_Video_0400 + VIC_CharS_1000 + VIC_CharS_unus] ; scrn=$0400-$07e7 chr=$1000-$17ff
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda #$c4 + VIC_MemBank_0    ; ##...### - $03 = $0000-$3fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda TabBackGroundColorSav   ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        lda #$34                    ; restore IRQ
                        sta C64_CINV_LO             ; 
                        
                        lda #$ea                    ; 
                        sta C64_CINV_HI             ; 
                        
                        ldx #$1b                    ; 
                        ldy #$41                    ; 
                        stx TIMALO                  ; CIA1($DC04) Timer A (low byte)
                        sty TIMAHI                  ; CIA1($DC05) Timer A (high byte)
                        
                        lda #$81                    ; #......# - enable timer A interrupts
                        sta CIAICR                  ; CIA1($DC0D) Interrupt Control
                        
                        lda CIACRA                  ; CIA1($DC0E) Control Register A
                        and #$80                    ; #.......
                        ora #$11                    ; ...#...# - timer A start
                        sta CIACRA                  ; CIA1($DC0E) Control Register A
                        
                        cli                         ; 
C64ResetX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitVoc1              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitVoc1                subroutine                  ; 
                        jsr ResetVoc1               ; 
                        
                        lda #$28                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        lda #$fb                    ; #####.##
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
InitVoc1X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitVoc3              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitVoc3                subroutine                  ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$20                    ; 
                        sta FRELO3                  ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        lda #$01                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$c8                    ; 
                        sta PWLO3                   ; SID($D410) Oscillator 3 Pulse Waveform Width (low byte)
                        lda #$09                    ; 
                        sta PWHI3                   ; SID($D411) Oscillator 3 Pulse Waveform Width (high nybble)
                        lda #$20                    ; 
                        sta SUREL3                  ; SID($D414) Oscillator 3 Sustain/Release
                        lda #$41                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
InitVoc3X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetSid              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetSid                subroutine                  ; 
                        ldx #$17                    ; 
                        lda #$00                    ; 
.SetInit                sta SID,x                   ; SID($D400)
                        dex                         ; 
                        bpl .SetInit                ; 
                        
                        lda #$1f                    ; 
                        sta SIGVOL                  ; SID($D418) Volume/Filter Select
                        
ResetSidX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc1             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc1               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.IniVoc1                sta FRELO1,y                ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .IniVoc1                ; 
                        
ResetVoc1X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc2             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc2               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.IniVoc2                sta FRELO2,y                ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .IniVoc2                ; 
                        
ResetVoc2X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetVoc3             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetVoc3               subroutine                  ; 
                        ldy #$06                    ; 
                        lda #$00                    ; 
.ResetVoc3              sta FRELO3,y                ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        dey                         ; 
                        bpl .ResetVoc3              ; 
                        
ResetVoc3X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetVoc2_Cannon        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SetVoc2_Cannon          subroutine                  ; 
                        jsr ResetVoc2               ; 
                        
                        lda TabVoc2AttDec           ; 
                        sta ATDCY2                  ; SID($D40C) Oscillator 2 Attack/Decay
                        lda TabVoc2FreqHi           ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        lda #$81                    ; 
                        sta VCREG2                  ; SID($D40B) Oscillator 2 Control
                        
SetVoc2_CannonX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetVoc3_Cannon        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SetVoc3_Cannon          subroutine                  ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$34                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$0a                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
SetVoc3_CannonX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankAdaptSfxWithPosY  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankAdaptSfxWithPosY    subroutine                  ; 
                        lda TabWaitFlagSfx          ; 
                        beq .Adapt                  ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.Adapt                  inc TabWaitFlagSfx          ; TabWaitFlag_Yes
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc #$68                    ; 
                        asl a                       ; 
                        sta FRELO2                  ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        
TankAdaptSfxWithPosYX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankSfxShot           Does    :  
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankSfxShot             subroutine                  ; 
                        lda TabWaitFlagSfxShot      ; 
                        beq .Shot                   ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.Shot                   jsr ResetVoc3               ; 
                        
                        lda #$0a                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$14                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        lda #$0a                    ; 
                        sta TabPlaySfxVoc1_06       ; 
                        
                        inc TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
TankSfxShotX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankSfxHit            Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankSfxHit              subroutine                  ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$0c                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$0a                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        lda #$3c                    ; 
                        sta TabPlaySfxVoc1_06       ; 
                        
                        inc TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
TankSfxHitX             rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEneBulletSfxIni Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEneBulletSfxIni  subroutine                  ; 
                        lda TabBulletActive         ; 
                        cmp #TabBulletActiveYes     ; 
                        beq .ChkSfxBeep             ; 
                        
.Exit                   rts                         ; 
                        
.ChkSfxBeep             lda #$20                    ; ..#.....
                        bit TabSfxToPlay            ; 
                        bne .Exit                   ; 
                        
                        lda TabPlaneDistance        ; 
                        cmp #$02                    ; 
                        bne .Exit                   ; not the correct distance
                        
.SetBeep                lda #$20                    ; ..#.....
                        ora TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        lda #$01                    ; 
                        sta TabPlaySfxVoc1_05       ; 
                        lda #$02                    ; 
                        sta TabPlaySfxVoc2_05       ; 
                        
                        inc TabWaitFlagSfx          ; TabWaitFlag_Yes
                        
                        jsr ResetVoc3               ; 
                        
                        lda #$f0                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$ff                    ; 
                        sta TabVoc3FreqHi           ; 
                        lda #$11                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
SeaShipEneBulletSfxIniX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEneBulletSfxMod Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEneBulletSfxMod  subroutine                  ; 
                        lda #$20                    ; ..#.....
                        bit TabSfxToPlay            ; 
                        bne .ChkWait                ; 
                        
.Exit                   rts                         ; 
                        
.ChkWait                lda TabWaitFlagSfx          ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSfx          ; TabWaitFlag_Yes
                        
.LowerTone              lda TabVoc3FreqHi           ; 
                        sec                         ; 
                        sbc #$01                    ; 
                        sta TabVoc3FreqHi           ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        
SeaShipEneBulletSfxModX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveSfxShipEngine     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveSfxShipEngine       subroutine                  ; 
                        lda TabCaveShipSpeed        ; 
                        asl a                       ; 
                        asl a                       ; 
                        asl a                       ; 
                        clc                         ; 
                        adc #$30                    ; 
                        sta FRELO3                  ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        lda #$00                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        
CaveSfxShipEngineX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneSoundTurn     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneSoundTurn       subroutine                  ; 
                        lda #$04                    ; .....#..
                        bit TabSfxToPlay            ; 
                        bne .ChkSfxActive           ; 
                        
.Exit                   rts                         ; 
                        
.ChkSfxActive           lda TabWaitFlagSfx          ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSfx          ; TabWaitFlag_Yes
                        
                        lda TabVoc1FreqLo           ; 
                        sec                         ; 
                        sbc #$02                    ; 
                        sta TabVoc1FreqLo           ; 
                        bcs .ChkFreq                ; greater/equal
                        
                        dec TabVoc3FreqHi           ; 
                        
.ChkFreq                lda TabVoc3FreqHi           ; 
                        cmp #$04                    ; 
                        bcs .SfxMod                 ; greater/equal
                        
.SfxOff                 lda #$fb                    ; #####.##
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        rts                         ; 
                        
.SfxMod                 sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        sec                         ; 
                        sbc #$04                    ; 
                        
                        clc                         ; 
                        ror a                       ; 
                        
                        lda TabVoc1FreqLo           ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        
                        ror a                       ; 
                        sta PWLO1                   ; SID($D402) Oscillator 1 Pulse Waveform Width (low byte)
                        
SeaPlaneSoundTurnX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneSoundShoot    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneSoundShoot      subroutine                  ; 
                        lda #$08                    ; ....#...
                        bit TabSfxToPlay            ; 
                        bne .ChkActive              ; 
                        
.Exit                   rts                         ; 
                        
.ChkActive              lda TabWaitFlagSfxShot      ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
                        lda TabWaitFlagSfxShotPlane ; 
                        beq .SfxOff                 ; TabWaitFlag_No
                        
.Shoot                  jsr ResetVoc3               ; 
                        
                        lda #$20                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$09                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
                        rts                         ; 
                        
.SfxOff                 lda #$f7                    ; ####.###
                        and TabSfxToPlay            ; 
                        ora #$01                    ; 
                        sta TabSfxToPlay            ; 
                        
                        jsr InitVoc3                ; 
SeaPlaneSoundShootX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipReduceCount    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipReduceCount      subroutine                  ; 
                        lda TabPlayerLives          ; 
                        cmp #$05                    ; max possible for tank chapter
                        bcs .Reduce                 ; greater/equal
                        
                        rts                         ; 
                        
.Reduce                 dec TabPlayerLives          ; 
                        jsr ScreenStatusShowShips   ; 
                        
                        lda #$14                    ; 
                        sta TabWrkValueSav          ; 
                        
.GoResetTOD1            jsr ResetTOD1               ; 
                        
.GetTOD1_10th           lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp #$01                    ; 
                        bcc .GetTOD1_10th           ; lower
                        
                        jsr AddScore_100            ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr ResetVoc1               ; 
                        jsr ResetVoc2               ; 
                        jsr ResetVoc3               ; 
                        
                        lda #$ff                    ; 
                        sta CUTHI                   ; SID($D416) Filter Cutoff Frequency (high byte)
                        lda #$3a                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$07                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        dec TabWrkValueSav          ; 
                        
                        lda TabWrkValueSav          ; 
                        bne .GoResetTOD1            ; 
                        
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
SeaShipReduceCountX     jmp SeaShipReduceCount      ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ChapterStartBell      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ChapterStartBell        subroutine                  ; 
                        lda #$02                    ; 
                        jsr ResetChapterBell        ; 
                        
                        lda #$02                    ; ......#.
                        ora TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #$03                    ; 
                        sta SIGVOL                  ; SID($D418) Volume/Filter Select
                        jsr ResetTOD1               ; 
                        
.GoRingBell             jsr RingChapterBell         ; 
                        
                        lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        cmp #$03                    ; 
                        bcc .GoRingBell             ; 
                        
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        
                        lda #$fd                    ; ######.#
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        jsr ResetChapterBell        ; 
ChapterStartBellX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; RingChapterBell       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
RingChapterBell         subroutine                  ; 
                        lda #$02                    ; ......#.
                        bit TabSfxToPlay            ; 
                        bne .ChkExit                ; 
                        
.Exit                   rts                         ; 
                        
.ChkExit                lda TabWaitFlagPing         ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagPing         ; TabWaitFlag_Yes
                        
                        lda #$1e                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #$00                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        lda #$0e                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$24                    ; 
                        sta FRELO3                  ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$6b                    ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        lda #$30                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$15                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
RingChapterBellX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetChapterBell      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetChapterBell        subroutine                  ; 
                        ldx #$01                    ; 
                        lda #$00                    ; 
.Reset                  sta FRELO1,x                ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        sta FRELO2,x                ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        sta FRELO3,x                ; SID($D40E) Oscillator 3 Freq Ctrl (low byte)
                        dex                         ; 
                        bpl .Reset                  ; 
                        
ResetChapterBellX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapBlinkTargetsPing   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapBlinkTargetsPing     subroutine                  ; 
                        lda TabWaitFlagAutoAction   ; 
                        beq .SetBlinkWait           ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.SetBlinkWait           inc TabWaitFlagAutoAction   ; TabWaitFlag_Yes
                        
                        lda BGCOL1                  ; VIC($D022) Background Color 1
                        and #$0f                    ; ....####
                        bne .GetBlinkBlack          ; 
                        
.GetBlinkYellow         lda #YELLOW                 ; 
                        sta BGCOL1                  ; VIC($D022) Background Color 1
                        
                        rts                         ; 
                        
.GetBlinkBlack          lda #BLACK                  ; 
                        sta BGCOL1                  ; VIC($D022) Background Color 1
                        
                        lda #$01                    ; .......#
                        bit TabSfxToPlay            ; 
                        bne .MapPing                ; 
                        
.Exit                   rts                         ; 
                        
.MapPing                lda TabWaitFlagPing         ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagPing         ; TabWaitFlag_Yes
                        
                        lda #$bb                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #$13                    ; 
                        sta SIGVOL                  ; SID($D418) Volume/Filter Select
                        lda #$00                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$20                    ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        lda #$60                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$15                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
.CheckFleetPos          lda TabMoveSpriteLeRiTime   ; 
                        ora TabMoveSpriteUpDoTime   ; 
                        beq .PingOff                ; match - no pings
                        
                        lda #$f0                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        
                        rts                         ; 
                        
.PingOff                lda #$00                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        
MapBlinkTargetsPingX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveSfxTorpedoShot    Does : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveSfxTorpedoShot      subroutine                  ; 
                        lda #$15                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$8a                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$00                    ; 
                        sta SUREL1                  ; SID($D406) Oscillator 1 Sustain/Release
                        ldy #$80                    ; 
                        sty VCREG1                  ; SID($D404) Oscillator 1 Control
                        iny                         ; 
                        sty VCREG1                  ; SID($D404) Oscillator 1 Control
                        
CaveSfxTorpedoShotX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneSoundPase05   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneSoundPase05     subroutine                  ; 
                        jsr ResetVoc1               ; 
                        
                        lda #$04                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$01                    ; 
                        sta PWHI1                   ; SID($D403) Oscillator 1 Pulse Waveform Width (high nybble)
                        lda #$bc                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$41                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        lda #$04                    ; .....#..
                        ora TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
__SeaPlaneSoundPase05   equ * + $01                 ; 
                        lda #$82                    ; 
                        sta TabVoc1FreqLo           ; 
                        
                        lda #$05                    ; 
                        sta TabVoc3FreqHi           ; 
                        
                        lda #$01                    ; 
                        sta TabPlaySfxVoc2_05       ; 
                        
SeaPlaneSoundPase05X    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlyPhases     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlyPhases       subroutine                  ; 
                        ldx #$07                    ; 
.GetNextSprite          lda TabSpriteStatus,x       ; 
                        bne .ChkActive              ; TabSpriteActiveYes/ TabSpriteActiveNo
                        
                        jsr SeaPlaneSpriteInit      ; TabSpriteStatusFree
                        
.ChkActive              cmp #TabSpriteActiveYes     ; 
                        beq .Active                 ; 
                        
.Inactive               jmp .SetNextSprite          ; 
                        
.Active                 lda TabPlaneDistance,x      ; 
                        bne .ChkDist_01             ; 
                        
                        jsr SeaPlaneFlySetPhase00   ; 
                        
.ChkDist_01             lda TabPlaneDistance,x      ; 
                        cmp #$01                    ; 
                        bne .ChkDist_02             ; 
                        
                        jsr SeaPlaneFlySetPhase01   ; 
                        
.ChkDist_02             lda TabPlaneDistance,x      ; 
                        cmp #$02                    ; 
                        bne .ChkDist_03             ; 
                        
                        jsr SeaPlaneFlySetPhase02   ; 
                        
.ChkDist_03             lda TabPlaneDistance,x      ; 
                        cmp #$03                    ; 
                        bne .ChkDist_04             ; 
                        
                        jsr SeaPlaneFlySetPhase03   ; 
                        
.ChkDist_04             lda TabPlaneDistance,x      ; 
                        cmp #$04                    ; 
                        bne .ChkDist_05             ; 
                        
                        jsr SeaPlaneFlySetPhase04   ; 
                        
.ChkDist_05             lda TabPlaneDistance,x      ; 
                        cmp #$05                    ; 
                        bne .SetNextSprite          ; 
                        
                        jsr SeaPlaneFlySetPhase05   ; 
                        
.SetNextSprite          dex                         ; 
                        cpx #$04                    ; 
                        bne .GetNextSprite          ; 
                        
SeaPlaneFlyPhasesX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneSpriteInit    Does    : Start from carrier - Fly left
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneSpriteInit      subroutine                  ; 
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda #BH_SprtStartPlanePosX  ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
.ClrPlanePosX_MSB       jsr SetAcToXrTimesMult2     ; 
                        eor #$ff                    ; ########
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda #BH_SprtStartPlanePosY  ; 
                        sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #BH_SprtColorPlane      ; 
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        
                        lda YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        and #$0f                    ; ....####
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        lda XXPAND                  ; VIC($D01D) Sprite Double Height
                        and #$0f                    ; ....####
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #SP_Ptr_Plane_00        ; plane single sprite phase 1
                        sta BH_SpritePointers_A,x   ; 
                        
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #TabSpriteActiveYes     ; 
                        sta TabSpriteStatus,x       ; 
                        
                        lda #TabSpriteActiveYes     ; 
                        sta TabSpriteActivePlane,x  ; 
                        
                        lda #TabMoveSprite_Le + $10 ; #..#.... - flag: move left + value
                        sta TabMoveSpriteLeRi,x     ; 
                        
                        lda #TabPlaneDirReLiCont    ; 
                        sta TabPlaneDirReLi,x       ; 
                        
                        lda #TabMoveSprite_Up + $02 ; ......#. - flag: move up + value
                        sta TabMoveSpriteUpDo,x     ; 
                        
                        lda #TabPlaneDirUpDo_1      ; 
                        sta TabPlaneDirUpDo,x       ; 
                        
                        lda #$00                    ; 
                        sta TabPlaneFlies,x         ; 
                        sta TabPlaneSpeedAppr,x     ; TabPlaneSpeedApprNone
                        
                        lda #$00                    ; 
                        sta GR_SpriteStepCount,x    ; 
                        
                        lda #$06                    ; 
                        sta TabWaitTimePlaneFlyLe,x ; 
                        
                        lda #TabPlaneDist_Ini       ; 
                        sta TabPlaneDistance,x      ; 
                        
SeaPlaneSpriteInitX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase00 Does    : Check Turn Point
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase00   subroutine                  ; 
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #BH_SprtTurnPlanePosX   ; 
                        bcs SeaPlaneFlySetPhase00X  ; greater/equal - not reached
                        
.StopAndTurn            lda #$00                    ; 
                        sta TabMoveSpriteLeRiTime,x ; 
                        
                        lda #TabPlaneDirReLi_3      ; 
                        sta TabPlaneDirReLi,x       ; 
                        
                        lda #TabPlaneSpeedApprSlow  ; 
                        sta TabPlaneSpeedAppr,x     ; 
                        
                        lda #TabPlaneFliesYes       ; 
                        sta TabPlaneFlies,x         ; 
                        
.SetNextPhase           lda #$01                    ; 
                        sta TabPlaneDistance,x      ; 
                        
SeaPlaneFlySetPhase00X  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase01 Does    : Climb and continue flying left
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase01   subroutine                  ; 
                        lda TabMoveSpriteLeRi,x     ; 
                        and #$7f                    ; .####### - ignore move left flag
.ChkValue               beq .Climb                  ; 
                        
                        rts                         ; 
                        
.Climb                  lda #TabPlaneDirReLi_4      ; 
                        sta TabPlaneDirReLi,x       ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        jsr GetRND                  ; 
                        
                        cmp #$28                    ; 
                        bcs .SubPosX                ; greater/equal
                        
                        adc #$38                    ; lower
                        
.SubPosX                sec                         ; 
                        sbc SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .Div4                   ; no underflow
                        
                        tay                         ; save
                        
                        lda #TabPlaneDirReLi_2      ; 
                        sta TabPlaneDirReLi,x       ; 
                        
                        tya                         ; restore and make positive
                        eor #$ff                    ; ########
                        adc #$01                    ; 
                        lsr a                       ; div 8
                        
.Div4                   lsr a                       ; 
                        lsr a                       ; 
                        sta TabMoveSpriteLeRiTime,x ; 
                        
                        lda #$01                    ; .......#
                        bit TabSfxToPlay            ; 
                        beq .RndRestore             ; 
                        
                        jsr GetRND                  ; 
                        sta __SeaPlaneSoundPase05   ; 
                        
.RndRestore             jsr GetRND_4a_93            ; 
                        sta TabRndSav               ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc TabRndSav               ; 
                        bcs .Div_8                  ; no underflow
                        
                        lda #TabMoveSprite_Up + $70 ; .###.... - flag: move up + value * 8
                        
.Div_8                  lsr a                       ; ..###....
                        lsr a                       ; ...###...
                        lsr a                       ; ....###..
                        sta TabMoveSpriteUpDo,x     ; 
                        
                        lda #TabPlaneSpeedApprFast  ; 
                        sta TabPlaneSpeedAppr,x     ; 
                        
.SetNextPhase           lda #$02                    ; 
                        sta TabPlaneDistance,x      ; 
                        
SeaPlaneFlySetPhase01X  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase02 Does    : Approach player as single sprite
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase02   subroutine                  ; 
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$05                    ; 
                        bcs .SetTime                ; greater/equal
                        
                        rts                         ; 
                        
.SetTime                lda #$0c                    ; 
                        sta TabMoveSpriteUpDoTime,x ; 
                        
                        lda #$1e                    ; 
                        sta TabWaitTimePlaneFlySav,x; 
                        
                        lda #TabPlaneDirUpDo_2      ; 
                        sta TabPlaneDirUpDo,x       ; 
                        
.SetNextPhase           lda #$03                    ; 
                        sta TabPlaneDistance,x      ; 
                        
                        lda #SP_Ptr_Plane_01        ; plane single sprite phase 2
                        sta BH_SpritePointers_A,x   ; 
                        
SeaPlaneFlySetPhase02X  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase03 Does    : Init/Move Second Plane Sprite as Sprite Number 04
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase03   subroutine                  ; 
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$0b                    ; $3c+$0B=$47
                        beq .IniSprt_04             ; 
                        
                        clc                         ; 
                        adc #SP_Ptr_Plane_07 - $0b  ; plane single sprite phase 3-7
                        sta BH_SpritePointers_A,x   ; 
                        
                        rts                         ; 
                        
.IniSprt_04             jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
.AddPosX_2ndHalf        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc #$0c                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .ClrMSB                 ; 

.SetMSB                 lda #$10                    ; ...#....
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .AddPosX_1stHalf        ; 
                        
.ClrMSB                 lda #$ef                    ; ###.####
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.AddPosX_1stHalf        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$0c                    ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .GetSprtPtr             ; 
                        
                        jsr SetAcToXrTimesMult2     ; 
                        eor MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.GetSprtPtr             lda #SP_Ptr_Plane_07        ; plane double sprite phase 1 left
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #SP_Ptr_Plane_08        ; plane double sprite phase 1 right
                        sta BH_SpritePointer04_A    ; 
                        
                        lda TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi + $04 ; 
                        
                        lda TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo + $04 ; 
                        
                        lda TabPlaneSpeedAppr,x     ; 
                        sta TabPlaneSpeedAppr + $04 ; 
                        
                        lda TabPlaneDirReLi,x       ; 
                        sta TabPlaneDirReLi + $04   ; 
                        
                        lda TabPlaneDirUpDo,x       ; 
                        sta TabPlaneDirUpDo + $04   ; 
                        
                        lda TabPlaneFlies,x         ; 
                        sta TabPlaneFlies + $04     ; 
                        
                        lda TabMoveSpriteLeRiTime,x ; 
                        sta TabMoveSpriteLeRiTime + $04 ; 
                        
                        lda TabMoveSpriteUpDoTime,x ; 
                        sta TabMoveSpriteUpDoTime + $04 ; 
                        
.SetNextPhase           lda #$04                    ; 
                        sta TabPlaneDistance,x      ; 
                        
                        lda #$10                    ; ...#....
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        jsr SeaPlaneMov2ndSprite    ; 
                        
                        lda #BH_SprtColorPlane      ; 
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
SeaPlaneFlySetPhase03X  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase04 Does    : Continue Flight / Fire / Start Break Off
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase04   subroutine                  ; 
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$10                    ; 
                        beq .BreakOffPlane          ; 
                        
                        asl a                       ; $20
                        clc                         ; 
                        adc #SP_Ptr_Plane_11 - $20  ; 
                        sta BH_SpritePointers_A,x   ; plane double sprite phase 2 left/right-phase 6 left/right
                        
                        adc #$01                    ; $21
                        sta BH_SpritePointer04_A    ; 
                        
                        jsr SeaPlaneMov2ndSprite    ; 
                        
.ChkShootDistance       lda GR_SpriteStepCount,x    ; 
                        cmp #$0d                    ; 
                        bne .Exit                   ; 
                        
.IniShoot               lda #$fe                    ; #######.
                        and TabSfxToPlay            ; 
                        ora #$08                    ; ....#...
                        sta TabSfxToPlay            ; 
                        
                        lda #$06                    ; 
                        sta TabPlaySfxVoc2_06       ; 
                        sta TabPlaySfxVoc1_06       ; 
                        
                        inc TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
                        lda #$40                    ; 
                        sta TabPlaySfxVoc1_04       ; 
                        sta TabWaitFlagSfxShotPlane ; TabWaitFlag_Yes + extra
                        
.Exit                   rts                         ; 
                        
.BreakOffPlane          jsr SeaPlaneSoundPase05     ; 
                        
.ChkTurnPointRight      lda SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$c0                    ; 
                        bcc .ChkTurnPointLeft       ; lower
                        
.TurnUpRight            lda #TabPlaneTurnUpRi       ; 
                        sta TabPlaneTurn            ; 
                        
                        lda #TabMoveSprite_Ri + $1e ; flag: move right + value
                        sta TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi + $04 ; 
                        
                        lda #TabMoveSprite_Up + $1e ; flag: move up + value
                        sta TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo + $04 ; 
                        
                        lda #TabPlaneDirReLiCont    ; 
                        sta TabPlaneDirReLi,x       ; 
                        sta TabPlaneDirReLi + $04   ; 
                        
                        lda #TabPlaneDirUpDo_1      ; 
                        sta TabPlaneDirUpDo,x       ; 
                        sta TabPlaneDirUpDo + $04   ; 
                        
                        lda #SP_Ptr_Plane_1d        ; plane turn away right phase 1 left
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #SP_Ptr_Plane_1e        ; plane turn away right phase 1 right
                        sta BH_SpritePointer04_A    ; 
                        jmp .SetNextPhase           ; 
                        
.ChkTurnPointLeft       lda SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$80                    ; 
                        bcc .TurnUpLeft             ; lower
                        
.TurnUp                 lda #TabPlaneTurnUp         ; 
                        sta TabPlaneTurn            ; 
                        
                        lda #TabMoveSprite_Up + $1e ; flag: move up + value
                        sta TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo + $04 ; 
                        
                        lda #TabPlaneDirUpDo_1      ; 
                        sta TabPlaneDirUpDo,x       ; 
                        sta TabPlaneDirUpDo + $04   ; 
                        
                        lda #SP_Ptr_Plane_11        ; 
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #SP_Ptr_Plane_12        ; 
                        sta BH_SpritePointer04_A    ; 
                        jmp .SetNextPhase           ; 
                        
.TurnUpLeft             lda #TabPlaneTurnUpLe       ; 
                        sta TabPlaneTurn            ; 

                        lda #TabMoveSprite_Le + $30 ; #.##....
                        sta TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi + $04 ; 
                        
                        lda #TabMoveSprite_Up + $1e ; flag: move up + value
                        sta TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo + $04 ; 
                        
                        sta TabPlaneDirReLi,x       ; 
                        sta TabPlaneDirReLi + $04   ; 
                        sta TabPlaneDirUpDo,x       ; 
                        sta TabPlaneDirUpDo + $04   ; 
                        
                        lda #SP_Ptr_Plane_17        ; plane turn away left phase 1 left
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #SP_Ptr_Plane_18        ; plane turn away left phase 1 right
                        sta BH_SpritePointer04_A    ; 
                        
.SetNextPhase           lda #$05                    ; 
                        sta TabPlaneDistance,x      ; 
                        
SeaPlaneFlySetPhase04X  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlySetPhase05 Does    : Finish Break Off
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlySetPhase05   subroutine                  ; 
                        lda TabMoveSpriteLeRi,x     ; 
                        bpl .UpRightChkMSB          ; TabMoveSprite_Ri
                        
.UpLeft                 jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
.ChkMinPosX             cmp #$02                    ; 
                        bcc .SpriteOff              ; lower
                        bcs .ChkMinPosY             ; higher/equal
                        
.UpRightChkMSB          jsr SetAcToXrTimesMult2     ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ChkMinPosY             ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$70                    ; 
                        bcs .SpriteOff              ; higher/equal
                        bcc .ChkMinPosY             ; lower
                        
.Chk1stSpritePtrUp      cmp #TabPlaneTurnUp         ; 
                        bne .Set1stSpritePtrUpLe    ; 
                        
.Set1stSpritePtrUp      lda GR_SpriteStepCount,x    ; $10
                        asl a                       ; *2
                        clc                         ; 
                        adc #SP_Ptr_Plane_11 - $20  ; plane turn away mid phase 1-3
                        sta BH_SpritePointers_A,x   ; 
                        jmp .Set2ndSpritePointer    ; 
                        
.Set1stSpritePtrUpLe    lda GR_SpriteStepCount,x    ; 
                        asl a                       ; 
                        clc                         ; 
                        adc #SP_Ptr_Plane_17 - $20  ; plane turn away left phase 1-3
                        sta BH_SpritePointers_A,x   ; 
                        
.Set2ndSpritePointer    clc                         ; 
                        adc #$01                    ; 
                        sta BH_SpritePointer04_A    ; 
                        
.GoMove2ndSprite        jsr SeaPlaneMov2ndSprite    ; 
.ExitMove2ndSprite      rts                         ; 
                        
.SpriteOff              jsr DisableSpriteNoFromXr   ; 
                        
                        lda #TabSpriteStatusExpl    ; 
                        sta TabSpriteStatus,x       ; 
                        sta TabPlaneDistance,x      ; 
                        
                        lda #TabPlaneDirReLiStop    ; 
                        sta TabPlaneDirReLi,x       ; 
                        sta TabPlaneDirUpDo,x       ; 
                        sta TabPlaneVal_1           ; 
                        sta TabPlaneVal_2           ; 
                        
                        lda #TabSpriteActiveNo      ; 
                        sta TabSpriteActivePlane,x  ; 
                        
                        lda #$ef                    ; ###.####
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
.ExitOff                rts                         ; 
                        
.ChkMinPosY             jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$28                    ; 
                        bcc .SpriteOff              ; lower
                        
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$13                    ; 
                        bcs .GoMove2ndSprite        ; higher/equal
                        
                        lda TabPlaneTurn            ; 
                        cmp #TabPlaneTurnUpRi       ; 
                        bne .Chk1stSpritePtrUp      ; 
                        
.Set1stSpritePtrUpRi    lda GR_SpriteStepCount,x    ; 
                        asl a                       ; 
                        clc                         ; 
                        adc #SP_Ptr_Plane_1d - $20  ; plane turn away right phase 1-3
                        sta BH_SpritePointers_A,x   ; 
                        
SeaPlaneFlySetPhase05X  jmp .Set2ndSpritePointer    ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneMov2ndSprite  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneMov2ndSprite    subroutine                  ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ClrBit4                ; 
                        
.SetBit4                lda #$10                    ; ...#....
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .SetMSB                 ; 
                        
.ClrBit4                lda #$ef                    ; ###.####
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.SetMSB                 sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
.OffsetRightSide        adc #$18                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .CpyPosX                ; 
                        
                        lda #$10                    ; ...#....
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.CpyPosX                lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
SeaPlaneMov2ndSpriteX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneFlyToPlayer   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneFlyToPlayer     subroutine                  ; 
                        ldx #$07                    ; 
.GetNextSpriteDirReLi   lda TabPlaneDirReLi,x       ; 
                        bne .ChkReLi_Cont           ; 
                        
                        lda #$00                    ; TabPlaneDirReLiStop
                        sta TabMoveSpriteLeRi,x     ; 
                        jmp .ChkUpDo_96b8_1         ; 
                        
.ChkReLi_Cont           cmp #TabPlaneDirReLiCont    ; 
                        bne .ChkReLi_02             ; 
                        
                        jmp .ChkUpDo_96b8_1         ; 
                        
.ChkReLi_02             cmp #TabPlaneDirReLi_2      ; 
                        bne .ChkReLi_03             ; 
                        
                        lda #TabMoveSprite_Le       ; #....... - flag: move left
                        ora TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi,x     ; 
                        
.ChkReLi_96b8_1         lda TabWaitFlags_Block02,x  ; 
                        bne .ChkUpDo_96b8_1         ; 
                        
                        lda TabMoveSpriteLeRi,x     ; 
                        and #$7f                    ; .#######
                        cmp TabMoveSpriteLeRiTime,x ; 
                        bcs .ChkUpDo_96b8_1         ; greater/equal
                        
                        inc TabMoveSpriteLeRi,x     ; lower
                        jmp .ChkUpDo_96b8_1         ; 
                        
.ChkReLi_03             cmp #TabPlaneDirReLi_3      ; 
                        bne .ChkReLi_04             ; 
                        
                        lda #TabMoveSprite_Le       ; #....... - flag: move left
                        ora TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi,x     ; 
                        
.ChkReLi_96b8_2         lda TabWaitFlags_Block02,x  ; 
                        bne .ChkUpDo_96b8_1         ; 
                        
                        lda TabMoveSpriteLeRi,x     ; 
                        and #$7f                    ; .#######
                        cmp #$00                    ; 
                        beq .ChkUpDo_96b8_1         ; equal
                        bcc .ChkUpDo_96b8_1         ; lower
                        
                        dec TabMoveSpriteLeRi,x     ; 
                        jmp .ChkUpDo_96b8_1         ; 
                        
.ChkReLi_04             cmp #TabPlaneDirReLi_4      ; 
                        bne .ChkReLi_05             ; 
                        
                        lda #$7f                    ; .#######
                        and TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi,x     ; 
                        jmp .ChkReLi_96b8_1         ; 
                        
.ChkReLi_05             cmp #TabPlaneDirReLi_5      ; 
                        bne .GetNextSpriteDirUpDo   ; 
                        
                        lda #$7f                    ; .#######
                        and TabMoveSpriteLeRi,x     ; 
                        sta TabMoveSpriteLeRi,x     ; 
                        jmp .ChkReLi_96b8_2         ; 
                        
.ChkUpDo_96b8_1         lda TabWaitFlags_Block02,x  ; 
                        bne .GetNextSpriteDirUpDo   ; 
                        
                        lda #$01                    ; 
                        sta TabWaitFlags_Block02,x  ; 
                        
.GetNextSpriteDirUpDo   lda TabPlaneDirUpDo,x       ; 
                        bne .ChkUpDo_01             ; 
                        
                        lda #$00                    ; TabMoveSpriteUpDo_None = no value
                        sta TabMoveSpriteUpDo,x     ; 
                        jmp .Get_96c8               ; 
                        
.ChkUpDo_01             cmp #$01                    ; 
                        bne .ChkUpDo_02             ; 
                        
                        jmp .Get_96c8               ; 
                        
.ChkUpDo_02             cmp #$02                    ; 
                        bne .ChkUpDo_03             ; 
                        
                        lda #$7f                    ; 
                        and TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo,x     ; 
                        
.ChkUpDo_96c8_1         lda TabWaitFlags_Block04,x  ; 
                        bne .Get_96c8               ; 
                        
                        lda TabMoveSpriteUpDo,x     ; 
                        and #$7f                    ; 
                        cmp TabMoveSpriteUpDoTime,x ; 
                        bcs .Get_96c8               ; greater/equal
                        
                        inc TabMoveSpriteUpDo,x     ; 
                        jmp .Get_96c8               ; 
                        
.ChkUpDo_03             cmp #$03                    ; 
                        bne .ChkUpDo_04             ; 
                        
                        lda #$7f                    ; .#######
                        and TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo,x     ; 
                        
.ChkUpDo_96c8_2         lda TabWaitFlags_Block04,x  ; 
                        bne .Get_96c8               ; 
                        
                        lda TabMoveSpriteUpDo,x     ; 
                        and #$7f                    ; .#######
                        cmp #$00                    ; 
                        beq .Get_96c8               ; equal
                        bcc .Get_96c8               ; lower
                        
                        dec TabMoveSpriteUpDo,x     ; 
                        jmp .Get_96c8               ; 
                        
.ChkUpDo_04             cmp #$04                    ; 
                        bne .ChkUpDo_05             ; 
                        
                        lda #TabMoveSprite_Do       ; #....... - flag: move down
                        ora TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo,x     ; 
                        jmp .ChkUpDo_96c8_1         ; 
                        
.ChkUpDo_05             cmp #$05                    ; 
                        bne .Get_9668               ; 
                        
                        lda #TabMoveSprite_Do       ; #....... - flag: move down
                        ora TabMoveSpriteUpDo,x     ; 
                        sta TabMoveSpriteUpDo,x     ; 
                        jmp .ChkUpDo_96c8_2         ; 
                        
.Get_96c8               lda TabWaitFlags_Block04,x  ; 
                        bne .Get_9668               ; 
                        
                        lda #$01                    ; 
                        sta TabWaitFlags_Block04,x  ; 
                        
.Get_9668               lda TabPlaneFlies,x         ; 
                        bne .SetNextSprite          ; TabPlaneFliesYes
                        
                        lda #TabPlaneSpeedApprNone  ; 
                        sta TabPlaneSpeedAppr,x     ; 
                        
.SetNextSprite          dex                         ; 
                        cpx #$04                    ; 
                        beq SeaPlaneFlyToPlayerX    ; 
                        
                        jmp .GetNextSpriteDirReLi   ; 
                        
SeaPlaneFlyToPlayerX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneStartNext     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneStartNext       subroutine                  ; 
                        ldx TabPlaneFreeSprite      ; 
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$05                    ; wait time between starts
                        bcs .StartPlanes            ; greater/equal
                        
                        rts                         ; 
                        
.StartPlanes            txa                         ; 
                        tay                         ; yr=xr
                        ldx #$05                    ; 
.FindFreeSprite         lda TabSpriteStatus,x       ; 
                        cmp #TabSpriteStatusExpl    ; 
                        beq .FoundFreeSprite        ; 
                        
.SetNextSprite          inx                         ; 
                        cpx #$08                    ; 
                        bne .FindFreeSprite         ; 
                        
                        rts                         ; 
                        
.FoundFreeSprite        stx TabPlaneFreeSprite      ; 
                        
                        lda TabPlaneTranspWait      ; 
                        beq .ChkTransport           ; 
                        
                        dec TabPlaneTranspWait      ; 
                        jmp .IniPlane               ; 
                        
.ChkTransport           cpx #$05                    ; 
                        bne .IniPlane               ; 
                        
.IniTransp              lda #$06                    ; 
                        sta TabPlaneTranspWait      ; 
                        sty TabPlaneFreeSprite      ; 
                        
                        lda #$00                    ; 
                        sta GR_SpriteStepCount,x    ; 
                        jsr SeaPlaneInitTransp      ; 
                        rts                         ; 
                        
.IniPlane               lda #TabSpriteStatusFree    ; 
                        sta TabSpriteStatus,x       ; 
                        
SeaPlaneStartNextX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneInitTransp    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneInitTransp      subroutine                  ; 
                        lda #TabPlaneTranspStatusUsed ; 
                        sta TabPlaneTranspStatus    ; 
                        
.ChooseHight            jsr GetRND_4a_93            ; 
                        sec                         ; .hbu002. 
                        sbc #$20                    ; .hbu002. - adjustment hight - $93 far too deep
                        sta SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        
.ChooseDir              lsr a                       ; 
                        bcs .FlyFromLeftToRight     ; 
                        
.FlyFromRightToLeft     lda #$20                    ; ..#..... - set MSB_5
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$58                    ; 
                        sta SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #SP_Ptr_Plane_4b        ; transporter fly left
                        sta BH_SpritePointer05_A    ; 
                        
                        lda #$28                    ; 
                        ora #TabMoveSprite_Le       ; #....... - flag: move left
                        
.Finish                 sta TabMoveTranspLeRi       ; 
                        
                        lda #$ff                    ; 
                        sta TabPlaneDirReLi + $05   ; 
                        
                        lda #TabTranspActiveYes     ; 
                        sta TabTranspActive         ; 
                        
                        lda #BH_SprtColorTransp     ; 
                        sta SP5COL                  ; VIC($D02C) Color Sprite 5
                        
                        ldx #$05                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$02                    ; 
                        sta TabTranspStatus         ; 
                        
.Exit                   rts                         ; 
                        
.FlyFromLeftToRight     lda #$df                    ; ##.#####
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$00                    ; 
                        sta SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #SP_Ptr_Plane_4c        ; transporter fly right
                        sta BH_SpritePointer05_A    ; 
                        
                        lda #TabMoveSprite_Ri + $28 ; #....... - flag: move right
SeaPlaneInitTranspX     jmp .Finish                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneTranspChkMax  Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneTranspChkMax    subroutine                  ; 
                        lda TabPlaneTranspStatus    ; 
                        bne .ChkDirTransport        ; TabPlaneTranspStatusFree
                        
.Exit                   rts                         ; 
                        
.ChkDirTransport        lda TabMoveTranspLeRi       ; 
                        bmi .ChkMinPosX             ; TabMoveSprite_Le
                        
.ChkMaxPosX             lda #$20                    ; ..#.....
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .Exit                   ; 
                        
                        lda SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$58                    ; 
                        bcc .Exit                   ; lower
                        
.DisableTransp          lda #TabTranspActiveNo      ; 
                        sta TabTranspActive         ; 
                        
                        ldx #$05                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda #TabSpriteStatusExpl    ; 
                        sta TabTranspStatus         ; 
                        
                        lda #TabPlaneTranspStatusFree ; 
                        sta TabPlaneTranspStatus    ; 
                        
                        rts                         ; 
                        
.ChkMinPosX             lda #$20                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .Exit                   ; 
                        
                        lda SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$04                    ; 
                        bcc .DisableTransp          ; 
                        
SeaPlaneTranspChkMaxX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneTranspChkHit  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneTranspChkHit    subroutine                  ;
                        lda TabPlaneTranspStatus    ; 
                        bne .ChkCollision           ; TabPlaneTranspStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkCollision           lda #$20                    ; ..#.....
                        bit TabSprtSprtCollReg      ; 
                        bne .ChkCollisionShotIni    ; 
                        
                        rts                         ; 
                        
.ChkCollisionShotIni    ldx #$02                    ; 
.ChkCollisionShot       jsr SetAcToXrTimesMult2     ; 
                        bit TabSprtSprtCollReg      ; ......#. and .......#
                        bne .ChkWaitTime            ; 
                        
.SetNextShot            dex                         ; 
                        bne .ChkCollisionShot       ; 
                        
                        rts                         ; 
                        
.ChkWaitTime            lda GR_SpriteStepCount,x    ; 
                        cmp #$10                    ; 
                        bcc .SetNextShot            ; 
                        
                        jsr SetYrToXrMult2          ; 
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        bcs .ChkDirektHit           ; greater/equal
                        
                        eor #$ff                    ; make positve
                        adc #$01                    ; 
                        
.ChkDirektHit           cmp #$0a                    ; 
                        bcs .SetNextShot            ; greater/equal
                        
.Explode                jsr DisableSpriteNoFromXr   ; 
                        
                        lda #TabSpriteStatusFree    ; 
                        sta TabSpriteStatus,x       ; 
                        sta TabTranspActive         ; TabTranspActiveNo
                        
                        lda #$20                    ; ..#.....
                        ora SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BH_SprtColorTranspExpl_MC0 ; 
                        sta SPMC0                   ; VIC($D025) Sprite Multicolor Register 0
                        
                        lda #BH_SprtColorTranspExpl_MC1 ; 
                        sta SPMC1                   ; VIC($D026) Sprite Multicolor Register 1
                        
                        lda #BH_SprtColorTranspExplode ; 
                        sta SP5COL                  ; VIC($D02C) Color Sprite 5
                        
                        lda #SP_Ptr_Plane_48        ; explode transporter
                        sta BH_SpritePointer05_A    ; 
                        
                        lda #$14                    ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagPing         ; 
                        
                        lda #TabPlaneTranspWasHitYes; 
                        sta TabPlaneTranspWasHit    ; 
                        
                        jsr InitVoc1                ; 
                        jsr AddScore_2000           ; 
SeaPlaneTranspChkHitX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneTranspChkExpl Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneTranspChkExpl   subroutine                  ; 
                        lda TabPlaneTranspWasHit    ; 
                        bne .ChkWaitTime            ; TabPlaneTranspWasHitYes
                        
.Exit                   rts                         ; 
                        
.ChkWaitTime            lda TabWaitFlagPing         ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        ldx #$05                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda #$df                    ; ##.#####
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BH_SprtColorTranspExpl_Off ; 
                        sta SP5COL                  ; VIC($D02C) Color Sprite 5
                        
                        lda #TabPlaneTranspStatusFree ; 
                        sta TabPlaneTranspStatus    ; 
                        sta TabPlaneTranspWasHit    ; 
                        
                        lda #TabSpriteStatusExpl    ; 
                        sta TabTranspStatus         ; 
                        
SeaPlaneTranspChkExplX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetRND                Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetRND                  subroutine                  ; 
                        lda #$01                    ; .......# - Start Timer B (1=start, 0=stop)
                        sta CIACRB                  ; CIA1($DC0F) Control Register B
                        
                        lda RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        eor TIMBLO                  ; CIA1($DC06) Timer B (low byte)
                        eor TIMBHI                  ; CIA1($DC07) Timer B (high byte)
                        adc TabRndSeed              ; 
                        sta TabRndSeed              ; 
                        
GetRNDX                 rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetRND_4a_93          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetRND_4a_93            subroutine                  ; 
                        jsr GetRND                  ; 
                        
.ChkLimit               cmp #$4a                    ; 
                        bcs .Add_21                 ; greater/equal
                        
                        adc #$4a                    ; lower
                        rts                         ; 
                        
.Add_21                 adc #$20                    ; 
GetRND_4a_93X           jmp .ChkLimit               ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SwitchPlayerAndValues Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SwitchPlayerAndValues   subroutine                  ; 
                        ldx TabActualPlayerNumber   ; 
                        jsr PlayerValuesSave        ; 
                        
                        lda TabNumberOfPlayers      ; 
                        cmp #$02                    ; 
                        beq .ChkPlayer_02           ; 
                        
.RestorePlayerValues    ldx TabActualPlayerNumber   ; 
                        jsr PlayerValuesRestore     ; 
                        
                        lda #$00                    ; single player mode - player #1
                        rts                         ; 
                        
.ChkPlayer_02           cpx #$01                    ; switch players
                        bne .SetPlayer_01           ; 
                        
                        ldx #$00                    ; two player mode - player #1
                        stx TabActualPlayerNumber   ; 
                        jmp .RestorePlayerValues    ; 
                        
.SetPlayer_01           ldx #$01                    ; two player mode - player #2
                        stx TabActualPlayerNumber   ; 
                        jsr PlayerValuesRestore     ; 
                        
                        lda #$01                    ; player #2
SwitchPlayerAndValuesX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitLevel Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitLevel   subroutine                  ; 
                        jsr ScreenStatusTextOutRow1 ; xr=input text offset  yr=output text offset (Level)
                        
                        lda TabSkillLevel           ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffLevel1 ; 
                        
                        lda #WHITE                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffLevel1 ; 
                        
ScreenStatusInitLevelX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerSkillLevelInit  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerSkillLevelInit    subroutine                  ; 
                        ldx TabSkillLevel           ; 
                        dex                         ; 
                        
                        lda GR_bed8,x               ; 
                        sta __CaveSkillTorpedo_01   ; 
                        
                        lda GR_bef4,x               ; 
                        sta __CaveSkillTorpedo_02   ; 
                        
                        lda GR_bedc,x               ; 
                        sta __CaveSkillTorpedo_03   ; 
                        
                        lda GR_bef8,x               ; 
                        sta __CaveSkillTorpedo_04   ; 
                        
                        lda GR_bee0,x               ; 
                        sta __CaveSkillTorpedo_05   ; 
                        
                        lda GR_befc,x               ; 
                        sta __CaveSkillTorpedo_06   ; 
                        
                        lda GR_bee4,x               ; 
                        sta __CaveSkillTorpedo_07   ; 
                        
                        lda GR_bee8,x               ; 
                        sta __CaveSkillTorpedo_08   ; 
                        
                        lda GR_beec,x               ; 
                        sta __CaveSkillTorpedo_09   ; 
                        
                        lda GR_bef0,x               ; 
                        sta __CaveSkillTorpedo_10   ; 
                        
                        lda GR_bf00,x               ; 
                        sta __SkillAmmoInc          ; 
                        
                        lda GR_bf04,x               ; 
                        sta __ShootPlanesSkill_02   ; 
                        
                        sec                         ; 
                        sbc #$14                    ; 
                        sta __ShootPlanesSkill_01   ; 
                        
                        lda GR_bf08,x               ; 
                        sta __ShootPlanesDamage     ; 
                        
                        lda GR_bf0c,x               ; 
                        sta __DistSeaShipSkill      ; 
                        
                        lda GR_bf10,x               ; 
                        sta __BeachSpriteLevel      ; 
                        
                        lda GR_bf14,x               ; 
                        sta __TankEnemyShoot        ; 
                        
                        lda GR_bf18,x               ; 
                        sta __HillCannonBulletFly   ; 
                        
PlayerSkillLevelInitX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_MoveSprites       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_MoveSprites         subroutine                  ; 
                        ldx #$07                    ; 
.GetNextSprite          lda TabSpriteActivePlane,x  ; 
                        bne .ChkMoveHoriz           ; TabSpriteActiveYes
                        
.SetNextSprite          dex                         ; TabSpriteActiveNo
                        bpl .GetNextSprite          ; 
                        
.Exit                   rts                         ; 
                        
.ChkMoveHoriz           lda TabMoveSpriteLeRi,x     ; 
                        and #$7f                    ; .#######
.ChkAmount              beq .ChkMoveVert            ; no amount found
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabPlaneWaitHori,x      ; 
                        sta TabPlaneWaitHori,x      ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .GetSpriteOff           ; 
                        
                        sta TabPlaneWaitHori,x      ; 
                        inc TabMoveSpritesStep      ; 
                        
.GetSpriteOff           jsr SetYrToXrMult2          ; yr = xr * 2
                        lda TabMoveSpriteLeRi,x     ; 
                        bmi .MoveLeft               ; TabMoveSprite_Le
                        
.MoveRight              lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .ChkMoveVert            ; 
                        
.SetSpriteInMask        jsr SetAcToXrTimesMult2     ; 
                        eor MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .ChkMoveVert            ; 
                        
.MoveLeft               lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .SetSpriteInMask        ; 
                        
.ChkMoveVert            lda TabMoveSpriteUpDo,x     ; 
                        and #$7f                    ; .#######
                        beq .ApproachPlanes         ; no amount found
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabPlaneWaitVert,x      ; 
                        sta TabPlaneWaitVert,x      ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .MoveUp                 ; 
                        
                        sta TabPlaneWaitVert,x      ; 
                        inc TabMoveSpritesStep      ; 
                        
.MoveUp                 jsr SetYrToXrMult2          ; yr = xr * 2
                        lda TabMoveSpriteUpDo,x     ; 
                        bmi .MoveDown               ; TabMoveSprite_Do
                        
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        bcs .SetUp                  ; 
                        
                        lda #$b8                    ; 
                        
.SetUp                  sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        jmp .ApproachPlanes         ; 
                        
.MoveDown               lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        cmp #$ba                    ; 
                        bcc .SetDown                ; lower
                        
                        lda #$00                    ; 
                        
.SetDown                sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)

.ApproachPlanes         lda TabPlaneSpeedAppr,x     ; 
                        and #$7f                    ; .#######
                        beq .GoSetNextSprite        ; TabPlaneSpeedApprNone
                        
                        jsr IRQ_GetSpriteMoveStep   ; 
                        
                        clc                         ; 
                        adc TabPlaneWaitAppr,x      ; 
                        sta TabPlaneWaitAppr,x      ; 
                        
                        sec                         ; 
                        sbc #TabMoveSpritesStepTime ; 
                        bcc .ChkSpeed               ; 
                        
                        sta TabPlaneWaitAppr,x      ; 
                        inc TabMoveSpritesStep      ; 
                        
.ChkSpeed               lda TabPlaneSpeedAppr,x     ; 
                        bpl .IncSpriteStepCount     ; 
                        
.DecSpriteStepCount     lda GR_SpriteStepCount,x    ; 
                        sec                         ; 
                        sbc TabMoveSpritesStep      ; 
                        sta GR_SpriteStepCount,x    ; 
                        
                        jmp .GoSetNextSprite        ; 
                        
.IncSpriteStepCount     lda GR_SpriteStepCount,x    ; 
                        clc                         ; 
                        adc TabMoveSpritesStep      ; 
                        sta GR_SpriteStepCount,x    ; 
                        
.GoSetNextSprite        jmp .SetNextSprite          ; 
                        
IRQ_MoveSpritesX        equ *                       ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_GetSpriteMoveStep Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_GetSpriteMoveStep   subroutine                  ; 
                        sec                         ; 
                        sbc #$3c                    ; 
                        bcc .StepNone               ; 
                        
.StepOne                ldy #$01                    ; 
                        sty TabMoveSpritesStep      ; 
                        
                        rts                         ; 
                        
.StepNone               clc                         ; restore old value 1st
                        adc #$3c                    ; 
                        
                        ldy #$00                    ; 
                        sty TabMoveSpritesStep      ; 
                        
IRQ_GetSpriteMoveStepX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; C64IrqScrollInit      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
C64IrqScrollInit        subroutine                  ; 
                        lda #$00                    ; 
                        sta CIACRA                  ; CIA1($DC0E) Control Register A
                        
                        lda #$7f                    ; .#######
                        sta CIAICR                  ; CIA1($DC0D) Interrupt Control
                        sta CI2ICR                  ; CIA2($DD0D) Interrupt Control Register
                        
                        lda #<IRQ                   ; 
                        sta C64_CINV_LO             ; 
                        lda #>IRQ                   ; 
                        sta C64_CINV_HI             ; 
                        
                        lda #TabIRQScreenPartStatus ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        jsr IRQ_SetScreenGfx        ; 
                        
                        sei                         ; 
                        jsr IRQ_ScrollSoft          ; 
                        
                        lda #$01                    ; 
                        sta IRQMASK                 ; VIC($D01A) IRQ Mask Register
                        cli                         ; 
                        
C64IrqScrollInitX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_SetScreenGfx      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_SetScreenGfx        subroutine                  ; 
                        lda #$c4 + VIC_MemBank_1    ; ##...##. - $02 = $4000-$7fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        ora #VIC_MCM_On             ; ...#.... - Multi Color Mode=on
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda TabVicMemControl        ; 
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda TabBackGroundColor      ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
IRQ_SetScreenGfxX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScrollSoft        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScrollSoft          subroutine                  ; 
                        lda SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        and #[VIC_SoftY_Clear & VIC_RasterHi_Off] ; $78 - .####... (isolate soft scroll bits)
                        ora TabVicScrollSoftPosY    ; 
                        sta SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        
.StatusScreenStart      lda #$cd                    ; 
                        sta RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #VIC_SoftX_Clear        ; $f8 - #####... (isolate soft scroll bits)
                        ora TabVicScrollSoftPosX    ; 
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda #TabScrnTankBayScrollFinYes ; 
                        sta TabScrnTankBayScrollFin ; 
                        
IRQ_ScrollSoftX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ                   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ                     subroutine                  ; 
                        lda #$01                    ; 
                        sta VICIRQ                  ; VIC($D019) Interrupt Flags - Latched flags cleared if set to 1
                        
                        lda TabIRQScreenPartFlag    ; 
                        bne IRQ_ScreenPartGfx       ; 
                        
.WaitIni                ldx #$05                    ; 
.Wait                   dex                         ; 
                        bne .Wait                   ; 
                        
                        jsr IRQ_ScreenPartStatus    ; 
                        
                        lda #TabIRQScreenPartGfx    ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        cli                         ; 
                        
                        jsr IRQ_Exec_Waits          ; 
                        jsr IRQ_MoveSprites         ; 
                        jsr IRQ_CheckJoystick       ; 
                        
IRQ_Return              subroutine                  ; 
                        pla                         ; 
                        tay                         ; 
                        pla                         ; 
                        tax                         ; 
                        pla                         ; 
                        
IRQX                    rti                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenTankColors  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenTankColors    subroutine                  ; 
                        cli                         ; 
                        
                        ldx #$04                    ; 
                        ldy #$e0                    ; 
.__GetColorHi           equ * + $02                 ; 
.GetColor               lda BH_Scrn_Stat_Save - $e0,y ; 
.__SetColorHi           equ * + $02                 ; 
                        sta COLORAM - $e0,y         ; 
                        iny                         ; 
                        bne .GetColor               ; 
                        
                        inc .__GetColorHi           ; 
                        inc .__SetColorHi           ; 
                        
                        dex                         ; 
                        bne .GetColor               ; 
                        
                        lda #>[BH_Scrn_Stat_Save - $e0] ; 
                        sta .__GetColorHi           ; 
                        lda #>[COLORAM - $e0]       ; 
                        sta .__SetColorHi           ; 
                        
                        lda #TabScreenTankColorNo   ; 
                        sta TabScreenTankColorFlag  ; 
                        
IRQ_ScreenTankColorsX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenPartGfx     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenPartGfx       subroutine                  ; 
                        jsr IRQ_SetScreenGfx        ; 
                        jsr IRQ_ScrollSoft          ; 
                        
                        lda #TabIRQScreenPartStatus ; 
                        sta TabIRQScreenPartFlag    ; 
                        
                        lda TabScreenTankColorFlag  ; 
                        beq IRQ_ScreenPartGfxX      ; TabScreenTankColorNo   
                        
                        jsr IRQ_ScreenTankColors    ; 
IRQ_ScreenPartGfxX      jmp IRQ_Return              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_CheckJoystick     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_CheckJoystick       subroutine                  ; 
                        lda CIAPRA                  ; Data Port Register A
                        eor #$ff                    ; flip joystick action - $00 if set to $01
                        and #$1f                    ; ...##### - mask out joystick action
                        beq IRQ_CheckJoystickX      ; was none
                        
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
                        lda #TabGameModeGame        ; interrupt demo on user action
                        sta TabGameMode             ; 
                        
                        lda #TabUserActionYes       ; 
                        sta TabUserAction           ; 
                        
IRQ_CheckJoystickX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_Exec_Waits        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_Exec_Waits          subroutine                  ; 
                        ldy #$28                    ; 5 blocks a 8 bytes
                        ldx #$00                    ; 
.GetNextFlag            lda TabWaitFlags,x          ; 
                        beq .SetNext                ; TabWaitMove_Yes
                        
                        dec TabWaitTimes,x          ; 
                        bne .SetNext                ; 
                        
                        lda #$00                    ; 
.ResetFlag              sta TabWaitFlags,x          ; TabWaitMove_Yes (has to be TabWaitTimes + $28)
                        
.RestoreTime            lda TabWaitTimesSav,x       ; 
                        sta TabWaitTimes,x          ; 
                        
.SetNext                inx                         ; 
                        dey                         ; 
                        bne .GetNextFlag            ; 
                        
IRQ_Exec_WaitsX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankScrollTerrain     Does    : Scroll tank screen
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankScrollTerrain       subroutine                  ; 
                        lda TabScrnTankBayScrollFin ; 
                        bne TankScrollTerrainX      ; TabScrnTankBayScrollFinNo   
                        
                        lda TabWaitFlagMoveLeRi     ; 
                        bne TankScrollTerrainX      ; TabWaitFlag_Yes
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveLeRi     ; 
                        
                        jsr TankScrollStrandSoft    ; 
                        
                        ldy #$07                    ; 
                        cpy TabTankScrollSoftPosX   ; 
                        bcs .SetVicScrollX          ; greater/equal
                        
                        jsr TankScrollStrandHard    ; 
                        
                        dec TabScrnTankBayScrollPos ; 
                        dec TabScrnTankBayScrollDec ; 
                        dec TabScrnTankBayCharNext  ; 
                        bpl .NextScrollPosX         ; 
                        
                        ldx #TabScrnTankBayCharNextIni ; 
                        stx TabScrnTankBayCharNext  ; 
                        
.NextScrollPosX         lda TabTankScrollSoftPosX   ; 
                        sec                         ; 
                        sbc #$08                    ; 
                        sta TabTankScrollSoftPosX   ; 
                        
.SetVicScrollX          lda TabTankScrollSoftPosX   ; 
                        sta TabVicScrollSoftPosX    ; 
                        
                        lda #TabScrnTankBayScrollFinNo ; 
                        sta TabScrnTankBayScrollFin ; 
                        
TankScrollTerrainX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankScrollStrandHard  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankScrollStrandHard    subroutine                  ; 
                        ldx TabScrnTankBayScrollPos ; 
                        jsr TankStrandAreaSetPtr    ; and work fields
                        
                        lda #TabScrnTankPosYSky     ; 
                        sta TabScrnTankPosY         ; 
                        
.SetNextScreenPtrs      ldx TabScrnTankPosY         ; 
                        jsr TankObstIniScrnInPtr    ; 
                        
                        ldx TabScrnTankPosY         ; 
                        jsr TankObstIniScrnOutPtr   ; 
                        
                        inc BLZ_ScrnRowPtrGfxOutLo  ; 
                        inc BLZ_ScrnRowPtrColoRamOutLo ; 
                        
                        ldy #TabScrnTankPosXMax     ; 38 = gfx screen max width
.ScrollGfxColoRam       lda (BLZ_ScrnRowPtrGfxIn),y ; 
                        sta (BLZ_ScrnRowPtrGfxOut),y; 
                        
                        lda (BLZ_ScrnRowPtrColoRamIn),y ; 
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
                        dey                         ; 
                        bpl .ScrollGfxColoRam       ; 
                        
                        lda TabScrnTankBayCharNext  ; 
                        sta TabScrnTankBayCharPtr   ; 
                        
                        lda #$00                    ; 
                        sta TabScrnTankPosX         ; 
                        
                        dec BLZ_ScrnRowPtrGfxOutLo  ; 
                        dec BLZ_ScrnRowPtrColoRamOutLo ; 
                        jsr TankObstPaint           ; 
                        
                        inc TabScrnTankPosY         ; 
                        
                        lda #TabScrnTankPosYMax     ; gfx screen max height
                        cmp TabScrnTankPosY         ; 
                        bne .SetNextScreenPtrs      ; 
                        
                        jsr TankScreenFlipFlop      ; 
TankScrollStrandHardX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankObstIniScrnInPtr  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankObstIniScrnInPtr    subroutine                  ; 
                        lda TabScrnRowPtrLo,x       ; 
                        sta BLZ_ScrnRowPtrGfxInLo   ; 
                        sta BLZ_ScrnRowPtrColoRamInLo ; 
                        
                        lda TabTankActiveScreen     ; 
                        beq .ScreenB                ; screen B
                        
.ScreenA                lda TabScrnRowPtrGfxAHi,x   ; 
                        sta BLZ_ScrnRowPtrGfxInHi   ; 
                        
                        lda TabScrnRowPtrColoRamHi,x ; 
                        sta BLZ_ScrnRowPtrColoRamInHi ; 
                        
                        rts                         ; 
                        
.ScreenB                lda TabScrnRowPtrGfxBHi,x   ; 
                        sta BLZ_ScrnRowPtrGfxInHi   ; 
                        
                        lda TabScrnRowPtrColoRamHi,x ; 
                        sta BLZ_ScrnRowPtrColoRamInHi ; 
                        
TankObstIniScrnInPtrX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankObstIniScrnOutPtr Does    : 
;                       Expects : xr=TabScrnTankPosY        
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankObstIniScrnOutPtr   subroutine                  ; 
                        lda TabScrnRowPtrLo,x       ; 
                        sta BLZ_ScrnRowPtrGfxOutLo  ; 
                        sta BLZ_ScrnRowPtrColoRamOutLo ; 
                        
                        lda TabTankActiveScreen     ; 
                        beq .ScreenA                ; 
                        
.ScreenB                lda TabScrnRowPtrGfxBHi,x   ; 
                        sta BLZ_ScrnRowPtrGfxOutHi  ; 
                        
                        lda TabScrnRowPtrSaveHi,x   ; 
                        sta BLZ_ScrnRowPtrColoRamOutHi ; 
                        
.Exit                   rts                         ; 
                        
.ScreenA                lda TabScrnRowPtrGfxAHi,x   ; 
                        sta BLZ_ScrnRowPtrGfxOutHi  ; 
                        
                        lda TabScrnRowPtrSaveHi,x   ; 
                        sta BLZ_ScrnRowPtrColoRamOutHi ; 
                        
TankObstIniScrnOutPtrX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankScreenFlipFlop    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankScreenFlipFlop      subroutine                  ; 
                        sei                         ; 
                        
                        lda TabTankActiveScreen     ; 
                        beq .SetScreenA             ; 
                        
.SetScreenB             lda TabVicMemControlSetB    ; 
                        sta TabVicMemControl        ; 
                        
                        lda #$00                    ; 
                        bpl .SetActiveScreenFlag    ; 
                        
.SetScreenA             lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; 
                        
                        lda #$01                    ; 
.SetActiveScreenFlag    sta TabTankActiveScreen     ; 
                        
                        lda #TabScreenTankColorYes  ; 
                        sta TabScreenTankColorFlag  ; 
                        
                        cli                         ; 
TankScreenFlipFlopX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankScrollStrandSoft  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankScrollStrandSoft    subroutine                  ; 
                        ldy #$06                    ; 
                        ldx #$01                    ; 
                        lda TabTankSprtScrollSpeed  ; speed
.FillTabSpriteAdapt     sta TabTankAdaptPosX,x      ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bpl .FillTabSpriteAdapt     ; 
                        
                        lda TabTankScrollSoftPosX   ; 
                        clc                         ; 
                        adc TabTankSprtScrollSpeed  ; speed
                        sta TabTankScrollSoftPosX   ; 
                        
TankScrollStrandSoftX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_ScreenPartStatus  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_ScreenPartStatus    subroutine                  ; 
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        and #[VIC_SoftX_Clear & VIC_MCM_Off | VIC_Cols40] ; $e8 - ###.#...
                        ora #$00                    ; 
                        tax                         ; 
                        
                        lda SCROLY                  ; VIC($D011) VIC Control Register 1 (Vert Fine Scroll)
                        and #[VIC_SoftY_Clear & VIC_RasterHi_Off] ; $78 - .####... (isolate soft scroll bits)
                        ora #$03                    ; ......##
                        tay                         ; 
                        
                        lda TabBackGroundColorSav   ; 
                        jsr IRQ_Wait                ; 
                        
                        stx SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        lda #[VIC_Video_0400 + VIC_CharS_1000 + VIC_CharS_unus] ; scrn=$0400-$07e7 chr=$1000-$17ff
                        sta VMCSB                   ; VIC($D018) VIC Chip Memory Control
                        
                        lda #$c4 + VIC_MemBank_0    ; ##...### - $03 = $0000-$3fff
                        sta CI2PRA                  ; CIA2($DD00) Data Port A - Bits 0-1 = VIC mem bank
                        
                        lda #$ff                    ; 
                        sta RASTER                  ; VIC($D012) Read : Current Raster Scan Line (Bit 8 in SCROLY = $D011)
                        
IRQ_ScreenPartStatusX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IRQ_Wait              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IRQ_Wait                subroutine                  ; 
                        pha                         ; 
                        txa                         ; 
                        
                        ldx #$21                    ; 
.Wait                   dex                         ; 
                        bne .Wait                   ; 
                        
                        tax                         ; 
                        pla                         ; 
                        
                        bit SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bit D6510                   ; 6510 On-Chip I/O Data Direction
                        
IRQ_WaitX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankStrandAreaSetPtr  Does    : Init obstacle data area pointer and work fields
;                       Expects : xr=offset to data pointers
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankStrandAreaSetPtr    subroutine                  ; 
                        lda TabTankScrollAreaPtrHi,x; 
                        bne .SetDataPtr             ; 
                        
.ClrDataPtr             sta BLZ_TankObstDataPtrHi   ; $00
                        
                        rts                         ; 
                        
.SetDataPtr             sta BLZ_TankObstDataPtrHi   ; 
                        
                        lda TabTankScrollAreaPtrLo,x; 
                        sta BLZ_TankObstDataPtrLo   ; 
                        
                        ldy #$00                    ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstPosY         ; 
                        
                        iny                         ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstChrNum       ; 
                        
                        iny                         ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstColor        ; 
                        sty TabTankObstOffNext      ; point to next data pointers
                        
TankStrandAreaSetPtrX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankObstPaint         Does    : 
;                       Expects : xr=TabScrnTankPosY        
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankObstPaint           subroutine                  ; 
                        lda BLZ_TankObstDataPtrHi   ; 
                        beq .ChkSkyMax              ; 
                        
                        cpx TabTankObstPosY         ; 
                        bne .ChkSkyMax              ; 
                        
                        ldy TabScrnTankPosX         ; 
                        lda TabTankObstChrNum       ; 
.Obstacle               sta (BLZ_ScrnRowPtrGfxOut),y; 
                        
                        lda TabTankObstColor        ; 
.ObstacleColor          sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
.ChkEndOfData           and #$80                    ; #....... - flag: end of data
                        beq .GetNextDataAreaSet     ; not set
                        
                        lda #$00                    ; done
                        sta TabTankObstPosY         ; 
                        
                        rts                         ; 
                        
.GetNextDataAreaSet     ldy TabTankObstOffNext      ; point to next data pointers
                        iny                         ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstPosY         ; 
                        
                        iny                         ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstChrNum       ; 
                        
                        iny                         ; 
                        lda (BLZ_TankObstDataPtr),y ; 
                        sta TabTankObstColor        ; 
                        
                        sty TabTankObstOffNext      ; point to next data pointers
                        
                        rts                         ; 
                        
.ChkSkyMax              cpx #TabScrnTankPosYTreeTop ; 
                        bcs .ChkTreeTop             ; greater/equal
                        
.SetSky                 lda #$01                    ; lower is sky
                        pha                         ; 
                        lda #CYAN                   ; 
                        pha                         ; 
                        jmp .PaintObjWithColor      ; 
                        
.ChkTreeTop             bne .ChkTreeLeaf            ; greater
                        
.SetTreeTop             lda TabScrnTankBayCharPtr   ; equal is tree top
                        clc                         ; 
                        adc #$02                    ; 
                        pha                         ; 
                        lda #[COLORAM_MCM_On | CYAN]; ....#.## - sky over tree tops
                        pha                         ; 
                        jmp .PaintObjWithColor      ; 
                        
.ChkTreeLeaf            cpx #TabScrnTankPosYTreeLeaf;
                        bne .ChkTreeTwigs           ; 
                        
.SetTreeLeaf            lda #$01                    ; BayShoreChar01
                        pha                         ; 
                        lda #GREEN                  ; 
                        pha                         ; 
                        jmp .PaintObjWithColor      ; 
                        
.ChkTreeTwigs           cpx #TabScrnTankPosYTreeTwig;
                        bne .ChkTreeTrunk           ; 
                        
.SetTreeTwigs           lda TabTankScrollAreaTabPtr ; 
                        cmp #$f7                    ; 
                        bcc .GetBayCharPtrTwig      ; 
                        
                        lda #$00                    ; strand
                        jmp .SetBayCharPtr          ; 
                        
.GetBayCharPtrTwig      lda TabScrnTankBayCharPtr   ; 
                        clc                         ; 
                        adc #$0b                    ; 
.SetBayCharPtr          pha                         ; 
                        lda #[COLORAM_MCM_On | BLACK] ; ....#... - tree trunks
                        pha                         ; 
                        jmp .PaintObjWithColor      ; 
                        
.ChkTreeTrunk           cpx #TabScrnTankPosYTreeTrunk ; 
                        bne .SetStrand              ; 
                        
.SetTreeTrunk           lda TabTankScrollAreaTabPtr ; 
                        cmp #$f7                    ; 
                        bcc .GetBayCharPtrTrunk     ; lower
                        
                        lda #$00                    ; 
                        jmp .SetBayCharPtr          ; 
                        
.GetBayCharPtrTrunk     lda TabScrnTankBayCharPtr   ; 
                        clc                         ; 
                        adc #$14                    ; 
                        jmp .SetBayCharPtr          ; 
                        
.SetStrand              lda #$00                    ; strand
                        pha                         ; 
                        lda #GREEN                  ; 
                        pha                         ; 
                        
.PaintObjWithColor      ldy TabScrnTankPosX         ; 
                        pla                         ; 
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        pla                         ; 
                        sta (BLZ_ScrnRowPtrGfxOut),y; 
                        
TankObstPaintX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
TankScreenShowStart     subroutine                  ; 
                        lda #$00                    ; 
                        sta TabScrnTankPosX         ; 
                        sta TabScrnTankBayCharPtr   ; 
                        
.IniActualPosY          lda #TabScrnTankPosYSky     ; 
                        sta TabScrnTankPosY         ; 
                        
                        ldx TabTankScrollAreaTabPtr ; 
                        jsr TankStrandAreaSetPtr    ; and work fields
                        
.PaintScrnPartAtPosY    ldx TabScrnTankPosY         ; 
                        jsr TankObstIniScrnOutPtr   ; 
                        jsr TankObstPaint           ; 
                        
                        inc TabScrnTankPosY         ; 
                        
                        lda #TabScrnTankPosYMax     ; 
                        cmp TabScrnTankPosY         ; 
                        bne .PaintScrnPartAtPosY    ; 
                        
                        inc TabScrnTankBayCharPtr   ; 
                        
                        lda #$09                    ; 
                        cmp TabScrnTankBayCharPtr   ; 
                        bne .SetNextAreaPosX        ; 
                        
                        lda #$00                    ; 
                        sta TabScrnTankBayCharPtr   ; 
                        
.SetNextAreaPosX        inc TabTankScrollAreaTabPtr ; 
                        inc TabScrnTankPosX         ; 
                        
                        lda #TabScrnGfxActualMaxX   ; 
                        cmp TabScrnTankPosX         ; 
                        bne .IniActualPosY          ; 
                        
                        jsr TankScreenFlipFlop      ; 
                        
TankScreenShowStartX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetTOD1             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetTOD1               subroutine                  ; 
                        lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TODMIN                  ; CIA1($DC0A) Time of Day Clock Minutes
                        sta TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        sta TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        
ResetTOD1X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; WaitTOD1_Sec          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
WaitTOD1_Sec            subroutine                  ; 
                        jsr ResetTOD1               ; 
                        
.WaitTOD                lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        ldx TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp TabWaitTimeTOD_Sec      ; 
                        bne .WaitTOD                ; 
                        
WaitTOD1X               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; WaitTOD1_10th         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
WaitTOD1_10th           subroutine                  ; 
                        jsr ResetTOD1               ; 
                        
.WaitTOD                lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp TabWaitTimeTOD_10th     ; 
                        bne .WaitTOD                ; 
                        
WaitTOD1_10thX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenGraphicInit     Does    : Init game screen data
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenGraphicInit       subroutine                  ; 
                        stx BLZ_ScrnRowPtrGfxInLo   ; establish data pointers ...
                        sty BLZ_ScrnRowPtrGfxInHi   ;     ... to data in
                        
                        lda #<BH_Scrn_Gfx_A         ; 
                        sta BLZ_ScrnRowPtrGfxOutLo  ;     ... to gfx out
                        sta BLZ_ScrnRowPtrColoRamOutLo ;  ... to color out
                        
                        lda #>BH_Scrn_Gfx_A         ; 
                        sta BLZ_ScrnRowPtrGfxOutHi  ; 
                        
                        lda #>COLORAM               ; 
                        sta BLZ_ScrnRowPtrColoRamOutHi ; 
                        
.GetNextHeaderByte_0    jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn - overread 1st byte
.GetNextHeaderByte_1    jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        sta TabBackGroundColor      ; 
                        
.GetNextHeaderByte_2    jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        sta BGCOL1                  ; VIC($D022) Background Color 1
                        
.GetNextHeaderByte_3    jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        sta BGCOL2                  ; VIC($D023) Background Color 2
                        
                        ldy #$00                    ; 
.GetNextSingleByte      jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        
                        cmp #$ff                    ; EOD - single bytes
                        beq .GetNextMultiByte       ; 
                        
                        sta (BLZ_ScrnRowPtrGfxOut),y; 
                        
                        jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
                        iny                         ; 
                        bne .GoGetNextSingleByte    ; 
                        
                        inc BLZ_ScrnRowPtrGfxOutHi  ; 
                        inc BLZ_ScrnRowPtrColoRamOutHi ; 
                        
.GoGetNextSingleByte    jmp .GetNextSingleByte      ; 
                        
.GetNextMultiByte       jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        
                        cmp #$ff                    ; 2nd EOD - done
                        bne .IniNextMultiByte       ; 
                        
                        jmp .GoScreenGraphicCharSet ; go and get character set
                        
.IniNextMultiByte       sta TabTankObstChrNum       ; gfx chr byte
                        
                        jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
.IniNextMultiColor      sta TabTankObstColor        ; gfx chr color
                        
.IniNextMultiCount      jsr ScreenGetNextDataByte   ; get next byte from BLZ_ScrnRowPtrGfxIn
                        tax                         ; gfx chr/color count
.SetNextMultiByte       lda TabTankObstChrNum       ; 
                        sta (BLZ_ScrnRowPtrGfxOut),y; 
                        
                        lda TabTankObstColor        ; 
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
                        iny                         ; 
                        bne .DecCount               ; 
                        
                        inc BLZ_ScrnRowPtrGfxOutHi  ; 
                        inc BLZ_ScrnRowPtrColoRamOutHi ; 
                        
.DecCount               dex                         ; 
                        bne .SetNextMultiByte       ; 
                        
                        jmp .GetNextSingleByte      ; 
                        
.GoScreenGraphicCharSet ldx BLZ_ScrnRowPtrGfxInLo   ; 
                        ldy BLZ_ScrnRowPtrGfxInHi   ; 
                        jsr ScreenGraphicCharSet    ; 
                        
ScreenGraphicInitX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenGetNextDataByte Does    : Get next byte
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenGetNextDataByte   subroutine                  ; 
                        ldx #$00                    ; 
                        lda (BLZ_ScrnRowPtrGfxIn,x) ; 
                        
                        inc BLZ_ScrnRowPtrGfxInLo   ; 
                        bne ScreenGetNextDataByteX  ; 
                        
                        inc BLZ_ScrnRowPtrGfxInHi   ; 
ScreenGetNextDataByteX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenGraphicCharSet  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenGraphicCharSet    subroutine                  ; 
                        stx BLZ_CharDataPtrInLo     ; 
                        sty BLZ_CharDataPtrInHi     ; 
                        
                        ldy #<BH_Scrn_GfxCharSets   ; $00
                        sty TabCharDataPtrOutHi     ; 
                        sty BLZ_CharDataPtrOutLo    ; 
                        
                        ldy #>BH_Scrn_GfxCharSets   ; 
                        sty BLZ_CharDataPtrOutHi    ; 
                        
                        ldy #$01                    ; 
                        lda (BLZ_CharDataPtrIn),y   ; 
                        sta TabCharDataPtrOutLo     ; 
                        
                        asl TabCharDataPtrOutLo     ; 
                        rol TabCharDataPtrOutHi     ; *2
                        asl TabCharDataPtrOutLo     ; 
                        rol TabCharDataPtrOutHi     ; *4
                        asl TabCharDataPtrOutLo     ; 
                        rol TabCharDataPtrOutHi     ; *8
                        
                        lda TabCharDataPtrOutLo     ; 
                        clc                         ; 
                        adc BLZ_CharDataPtrOutLo    ; 
                        sta BLZ_CharDataPtrOutLo    ; 
                        
                        lda TabCharDataPtrOutHi     ; 
                        clc                         ; 
                        adc BLZ_CharDataPtrOutHi    ; 
                        sta BLZ_CharDataPtrOutHi    ; 
                        
                        ldy #$00                    ; 
                        lda (BLZ_CharDataPtrIn),y   ; 
                        sta TabSaveCharDataByte     ; 
                        
                        inc BLZ_CharDataPtrInLo     ; 
                        inc BLZ_CharDataPtrInLo     ; 
                        
                        ldx #$08                    ; 
                        ldy #$00                    ; 
.GetNextCharDataByte    lda (BLZ_CharDataPtrIn),y   ; 
                        sta (BLZ_CharDataPtrOut),y  ; 
                        dex                         ; 
                        bne .SetNextCharDataByte    ; 
                        
                        ldx #$08                    ; 
                        dec TabSaveCharDataByte     ; 
                        bne .SetNextCharDataByte    ; 
                        
                        rts                         ; 
                        
.SetNextCharDataByte    iny                         ; 
                        bne .GetNextCharDataByte    ; 
                        
                        inc BLZ_CharDataPtrInHi     ; 
                        inc BLZ_CharDataPtrOutHi    ; 
ScreenGraphicCharSetX   jmp .GetNextCharDataByte    ; 
; -------------------------------------------------------------------------------------------------------------- ;
; EnableSpriteNoFromXr  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
EnableSpriteNoFromXr    subroutine                  ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
EnableSpriteNoFromXrX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; DisableSpriteNoFromXr Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
DisableSpriteNoFromXr   subroutine                  ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        eor #$ff                    ; 
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
DisableSpriteNoFromXrX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetAcToXrTimesMult2   Does    : Shift ACC=$01 XR times left
;                       Expects : XR = count
;                       Returns : AC power of 2 - XR not changed
; -------------------------------------------------------------------------------------------------------------- ;
SetAcToXrTimesMult2     subroutine                  ; 
                        txa                         ; ac=xr
                        tay                         ; yr=xr=ac
                        
                        lda #$01                    ; 
                        
.NextPowerOfTwo         dey                         ; 
                        bmi SetAcToXrTimesMult2X    ; 
                        
                        asl a                       ; 
                        jmp .NextPowerOfTwo         ; 
                        
SetAcToXrTimesMult2X    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SetYrToXrMult2        Does    : yr = xr * 2
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SetYrToXrMult2          subroutine                  ; 
                        txa                         ; 
                        asl a                       ; 
                        tay                         ; 
                        
SetYrToXrMult2X         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpriteDataCopyInit    Does    : 
;                       Expects : xr=chapter number
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpriteDataCopyInit      subroutine                  ; 
                        lda #$1f                    ; 
                        sta SIGVOL                  ; SID($D418) Volume/Filter Select
                        
                        lda #$f7                    ; 
                        sta RESON                   ; SID($D417) Filter Resonance Ctrl
                        
                        lda TabSetFilterCutHi,x     ; 
                        sta CUTHI                   ; SID($D416) Filter Cutoff Frequency (high byte)
                        
                        lda TabSetFilterCutHi_5a    ; 
                        sta TabFilterCutHi          ; 
                        
                        lda #<BH_SpriteStore        ; 
                        sta BLZ_SpriteDataPtrOutLo  ; 
                        lda #>BH_SpriteStore        ; 
                        sta BLZ_SpriteDataPtrOutHi  ; target=$5000
                        
.ChkTyp_01_Bay          cpx #BH_ChapterBayAreaMap   ; 
                        bne .ChkTyp_02_Cave         ; 
                        
                        ldx #$00                    ; Map: Bay Area
.GetSprtPtrMapHi        lda SP_SpriteDatPtrHi_Map,x ; 
                        bne .SetSprtPtrMapHi        ; 
                        
                        rts                         ; 
                        
.SetSprtPtrMapHi        sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrMapLo        lda SP_SpriteDatPtrLo_Map,x ; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
                        jmp .GetSprtPtrMapHi        ; 
                        
.ChkTyp_02_Cave         cpx #BH_ChapterCaveCrossings; 
                        bne .ChkTyp_03_Planes       ; 
                        
                        ldx #$00                    ; Chapter: Cave Crossings
.GetSprtPtrCaveHi       lda SP_SpriteDatPtrHi_Cave,x; 
                        bne .SetSprtPtrCaveHi       ; 
                        
                        rts                         ; 
                        
.SetSprtPtrCaveHi       sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrCaveLo       lda SP_SpriteDatPtrLo_Cave,x; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
                        jmp .GetSprtPtrCaveHi       ; 
                        
.ChkTyp_03_Planes       cpx #BH_ChapterPlaneShootOut; 
                        bne .ChkTyp_04_Ships        ; 
                        
                        ldx #$00                    ; Chapter: Planes Shootout
.GetSprtPtrPlanesHi     lda SP_SpriteDatPtrHi_Plane,x ; 
                        bne .SetSprtPtrPlanesHi     ; 
                        
                        rts                         ; 
                        
.SetSprtPtrPlanesHi     sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrPlanesLo     lda SP_SpriteDatPtrLo_Plane,x ; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
                        jmp .GetSprtPtrPlanesHi     ; 
                        
.ChkTyp_04_Ships        cpx #BH_ChapterShipShootOut ; 
                        bne .ChkTyp_05_Beach        ; 
                        
                        ldx #$00                    ; Chapter: Ships Shootout
.GetSprtPtrShipsHi      lda SP_SpriteDatPtrHi_Ship,x; 
                        bne .SetSprtPtrShipsHi      ; 
                        
                        rts                         ; 
                        
.SetSprtPtrShipsHi      sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrShipsLo      lda SP_SpriteDatPtrLo_Ship,x; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
                        jmp .GetSprtPtrShipsHi      ; 
                        
.ChkTyp_05_Beach        cpx #BH_ChapterTanksOnBeach ; 
                        bne .SetTyp_06_Hill         ; 
                        
                        ldx #$00                    ; Chapter: Tanks on the Beach
.GetSprtPtrIntrosHi     lda SP_SpriteDatPtrHi_Beach,x ; 
                        bne .SetSprtPtrIntrosHi     ; 
                        
                        rts                         ; 
                        
.SetSprtPtrIntrosHi     sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrIntrosLo     lda SP_SpriteDatPtrLo_Beach,x ; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
                        jmp .GetSprtPtrIntrosHi     ; 
                        
.SetTyp_06_Hill         ldx #$00                    ; Chapter: Storm the Hill
.GetSprtPtrHillHi       lda SP_SpriteDatPtrHi_Hill,x; 
                        bne .SetSprtPtrHillHi       ; 
                        
                        rts                         ; 
                        
.SetSprtPtrHillHi       sta BLZ_SpriteDataPtrInHi   ; 
.GetSprtPtrHillLo       lda SP_SpriteDatPtrLo_Hill,x; 
                        jsr SpriteDataCopy          ; 
                        
                        inx                         ; 
SpriteDataCopyInitX     jmp .GetSprtPtrHillHi       ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpriteDataCopy        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpriteDataCopy          subroutine                  ; 
                        sta BLZ_SpriteDataPtrInLo   ; 
.SaveXr                 stx TabSaveSpriteDataCopyXR ; 
                        
                        ldy #$00                    ; 
                        lda (BLZ_SpriteDataPtrIn),y ; 
                        and #$40                    ; .#......
                        beq .CpyFullSprtDataI       ; 
                        
.CpyPartSprtDataI       ldx #$40                    ; max len
                        ldy #$00                    ; 
                        lda #$00                    ; 
.ClrPartSprtArea        sta (BLZ_SpriteDataPtrOut),y; 
                        
                        iny                         ; 
                        dex                         ; 
                        bne .ClrPartSprtArea        ; 
                        
                        jsr IncSprtDataPtr          ; 
                        
.CpyPartSprtData        lda (BLZ_SpriteDataPtrIn,x) ; 
                        cmp #$ff                    ; end of data
                        beq .PtrSetNextSprtTarget   ; 
                        
                        tay                         ; 
                        jsr IncSprtDataPtr          ; 
                        
                        lda (BLZ_SpriteDataPtrIn,x) ; 
                        sta (BLZ_SpriteDataPtrOut),y; 
                        jsr IncSprtDataPtr          ; 
                        
                        jmp .CpyPartSprtData        ; 
                        
.PtrSetNextSprtTarget   lda BLZ_SpriteDataPtrOutLo  ; 
                        clc                         ; 
                        adc #$40                    ; 
                        sta BLZ_SpriteDataPtrOutLo  ; 
                        bcc .RestoreXr              ; 
                        inc BLZ_SpriteDataPtrOutHi  ; 
                        
.RestoreXr              ldx TabSaveSpriteDataCopyXR ; 
.Exit                   rts                         ; 
                        
.CpyFullSprtDataI       jsr IncSprtDataPtr          ; 
                        
                        ldy #$00                    ; 
                        ldx #$3f                    ; 
.CpyFullSprtData        lda (BLZ_SpriteDataPtrIn),y ; 
                        sta (BLZ_SpriteDataPtrOut),y; 
                        iny                         ; 
                        dex                         ; 
                        bne .CpyFullSprtData        ; 
                        
                        lda BLZ_SpriteDataPtrInLo   ; 
                        clc                         ; 
                        adc #$3f                    ; 
                        sta BLZ_SpriteDataPtrInLo   ; 
                        bcc .PtrSetNextSprtTarget   ; 
                        inc BLZ_SpriteDataPtrInHi   ; 
                        
SpriteDataCopyX         jmp .PtrSetNextSprtTarget   ; 
; -------------------------------------------------------------------------------------------------------------- ;
; IncSprtDataPtr        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
IncSprtDataPtr          subroutine                  ; 
                        inc BLZ_SpriteDataPtrInLo   ; 
                        bne IncSprtDataPtrX         ; 
                        
                        inc BLZ_SpriteDataPtrInHi   ; 
IncSprtDataPtrX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitColor Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitColor   subroutine                  ; 
                        ldx #$00                    ; 
                        ldy #BH_Scrn_Stat_Rows * BH_Scrn_RowLen ; $c8                    ; 
.GetNextBlank           lda #$20                    ; " "
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01,x ; 
                        
                        lda #BH_StatColorStatusBase ; 
                        sta COLORAM + BH_Scrn_Stat_Row_01,x ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bne .GetNextBlank           ; 
                        
ScreenStatusInitColorX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitGfx   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitGfx     subroutine                  ; 
                        lda #$75                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 ; 
                        
                        lda #$fc                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 + $01 ; 
                        
                        lda #$a0                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 + BH_Scrn_RowLen - $01 ; 
                        
                        lda #$6f                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 + $02  ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 + BH_Scrn_RowLen - $03 ; 
                        
                        lda #$76                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_RowLen - $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_RowLen - $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_RowLen - $01 ; 
                        
                        lda #$fe                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_RowLen - $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_05 + BH_Scrn_RowLen - $02 ; 
                        
.Box                    lda #$ba                    ; 
.Row1Top                sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_OffBox + $00 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_OffBox + $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_OffBox + $02 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01 + BH_Scrn_Stat_OffBox + $03 ; 
                        
.Row2Left               sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_OffBox + $00 ; 
.Row2Right              sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_OffBox + $03 ; 
.Row3Left               sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_OffBox + $00 ; 
.Row3Right              sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_OffBox + $03 ; 
                        
                        lda #$a0                    ; 
.Row2CenterLeft         sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_OffBox + $01 ; 
.Row2CenterRight        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_02 + BH_Scrn_Stat_OffBox + $02 ; 
                        
                        lda #$9d                    ; 
.Row3BotRight           sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_OffBox + $01 ; 
                        
                        lda #$9b                    ; 
.Row3BotLeft            sta BH_Scrn_Stat + BH_Scrn_Stat_Row_03 + BH_Scrn_Stat_OffBox + $02 ; 
                        
                        lda #$63                    ; 
.Row4Add                sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_OffBox + $01 ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_OffBox + $02 ; 
                        
                        lda #BH_StatColorStatusBoxAdd ; 
                        sta COLORAM + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_OffBox + $01 ; 
                        sta COLORAM + BH_Scrn_Stat_Row_04 + BH_Scrn_Stat_OffBox + $02 ; 
                        
ScreenStatusInitGfxX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitMap   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitMap     subroutine                  ; 
                        lda #BH_StatColorLevel      ; 
                        sta TabScrnStatusTextColor  ; 

                        ldy #BH_Scrn_Stat_OffLevelTxt ; 
                        ldx #<TabTextStatusLevel    ; 
                        jsr ScreenStatusInitLevel   ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoTxt ; 
                        ldx #<TabTextStatusAmmo     ; 
                        jsr ScreenStatusTextOutRow1 ; xr=input text offset  yr=output text offset
                        
                        ldy #BH_Scrn_Stat_OffShipsTxt ; 
                        ldx #<TabTextStatusShips    ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorPlayer     ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffPlayerTxt ; 
                        ldx #<TabTextStatusPlayer   ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset (Player)
                        
                        ldx TabActualPlayerNumber   ; 
                        inx                         ; 
                        txa                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffPlayer1 ; 
                        
                        lda #BH_StatColorPlayer     ; 
                        sta COLORAM + BH_Scrn_Stat_OffPlayer1 ; 
                        
                        lda #BH_StatColorScore      ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffScoreTxt; 
                        ldx #<TabTextStatusScore    ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
ScreenStatusInitMapX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; AddScores             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
AddScore_100            subroutine                  ; 
                        lda #$01                    ; 
                        bne AddScores               ; always
                        
AddScore_200            subroutine                  ; 
                        lda #$02                    ; 
                        bne AddScores               ; always
                        
AddScore_400            subroutine                  ; 
                        lda #$04                    ; 
                        bne AddScores               ; always
                        
AddScore_600            subroutine                  ; 
                        lda #$06                    ; 
                        bne AddScores               ; always
                        
AddScore_800            subroutine                  ; 
                        lda #$08                    ; 
                        bne AddScores               ; always
                        
AddScore_1000           subroutine                  ; 
                        lda #$10                    ; 
                        bne AddScores               ; always
                        
AddScore_2000           subroutine                  ; 
                        lda #$20                    ; 
                        bne AddScores               ; always
                        
AddScore_9000           subroutine                  ; 
                        lda #$90                    ; 
                        
AddScores               subroutine                  ; 
.DecimalOn              sed                         ; 
                        
                        clc                         ; 
                        adc TabPlayerScoreLo        ; 
                        sta TabPlayerScoreLo        ; 
                        bcc .DecimalOff             ; 
                        
                        lda #$00                    ; 
                        adc TabPlayerScoreHi        ; 
                        sta TabPlayerScoreHi        ; 
                        
.DecimalOff             cld                         ; 
AddScoresX              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowScore Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowScore   subroutine                  ; 
                        lda TabPlayerScoreHi        ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        cpx #$30                    ; "0"
                        bne .SetScores              ; 
                        
                        ldx #$20                    ; " "
                        
.SetScores              stx BH_Scrn_Stat + BH_Scrn_Stat_OffScore100000 ; $0378
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffScore10000  ; $0379
                        
                        lda TabPlayerScoreLo        ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        stx BH_Scrn_Stat + BH_Scrn_Stat_OffScore1000   ; $037a
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffScore100    ; $037b
                        
                        lda #$30                    ; "0"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffScore10     ; $037c
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffScore1      ; $037d
                        
                        ldx #$00                    ; 
                        lda #BH_StatColorScore      ; 
                        ldy #BH_Scrn_Stat_LenScore; 
.SetScoreColor          sta COLORAM + BH_Scrn_Stat_OffScore100000,x ; 
                        inx                         ; 
                        dey                         ; 
                        bne .SetScoreColor          ; 
                        
ScreenStatusShowScoreX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MakeDigit             Does    : 
;                       Expects : ac=digit
;                       Returns : xr=right nibble yr=left nibble
; -------------------------------------------------------------------------------------------------------------- ;
MakeDigit               subroutine                  ; 
                        pha                         ; save
                        and #$0f                    ; ....####
                        
                        clc                         ; 
                        adc #$30                    ; make digit
                        tay                         ; 
                        
                        pla                         ; restore
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        
                        clc                         ; 
                        adc #$30                    ; make digit
                        tax                         ; 
                        
MakeDigitX              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGamePause        Does    :           ; on C= commodore key
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGamePause          subroutine                  ; 
                        lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; ########
                        and #CIA_KeyResCol_05       ; C= commodore key
                        bne .IniMerge               ; pressed
                        
                        rts                         ; 
                        
.IniMerge               ldy #$07                    ; 
                        lda #$00                    ; 
.MergeLoop              clc                         ; 
                        ldx TabSpriteActiveAll,y    ; 
                        beq .MrgStoreSpritesActive  ; 
                        
                        sec                         ; 
                        
.MrgStoreSpritesActive  rol a                       ; merge store into one byte
                        
                        dey                         ; 
                        bpl .MergeLoop              ; 
                        
.SavStoreSpritesActive  tax                         ; save merged ac
                        
                        lda #$00                    ; 
.ClrStoreSpritesActive  jsr AcToTabSpritesActive    ; 
                        
                        lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        jsr CheckGamePauseDelay     ; 
                        
.PauseGame              lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; ########
                        and #CIA_KeyResCol_05       ; C= commodore key
                        beq .PauseGame              ; 
                        
                        jsr CheckGamePauseDelay     ; 
                        
                        txa                         ; restore merged ac
.RestoreStoreAC_9789    jsr AcToTabSpritesActive    ; 
                        
                        lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        sta TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        
                        lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        sta TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        
CheckGamePauseX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGamePauseDelay   Does    : Wait time after GAME PAUSE to ready the controller again
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGamePauseDelay     subroutine                  ; 
                        lda #$30                    ; 
                        sta TabWaitTimePause        ; 
                        
.IniWait                lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagPause        ; 
                        
.Delay                  lda TabWaitFlagPause        ; IRQ_Exec_Waits
                        bne .Delay                  ; 
                        
CheckGamePauseDelayX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapChkFinished        Does    : Returns to start screen if idle too long or dead or move left on JoyStick 1
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapChkFinished          subroutine                  ; 
                        lda TabGameMode             ; 
                        bmi .TwoLevelReturn         ; TabGameModeExit
                        
                        lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; 
                        and #CIA_Joy_Le             ; 
                        bne .TwoLevelReturn         ; 
                        
                        rts                         ; 
                        
.TwoLevelReturn         pla                         ; level 1 - return to game start screen
                        pla                         ; 
                        
                        lda #TabShowScoresYes       ; 
                        sta TabPlayerShowScores     ; 
                        
MapChkFinishedX         rts                         ; now return to BASIC via GameStart
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGameFinished     Does    : Returns to start screen if idle too long or dead or move left on JoyStick 1
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGameFinished       subroutine                  ; 
                        lda TabGameMode             ; 
                        bmi .ThreeLevelReturn       ; TabGameModeExit
                        
                        lda CIAPRB                  ; CIA1($DC01) Data Port Register B
                        eor #$ff                    ; 
                        and #CIA_Joy_Le             ; 
                        bne .ThreeLevelReturn       ; 
                        
                        rts                         ; 
                        
.ThreeLevelReturn       pla                         ; level 1
                        pla                         ; 
                        
                        pla                         ; level 2 - return to game start screen
                        pla                         ; 
                        
                        lda #TabShowScoresYes       ; 
                        sta TabPlayerShowScores     ; 
                        
CheckGameFinishedX      rts                         ; now return to BASIC via GameStart
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusGetReady  Does    : Display 'get ready' screen
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusGetReady    subroutine                  ; 
                        jsr ResetSid                ; 
                        
                        lda #$00                    ; 
                        sta TabSfxToPlay            ; 
                        
                        lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; SetA
                        
                        lda #BLACK                  ; 
                        sta TabBackGroundColor      ; 
                        
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #$01                    ; screen A
                        sta TabTankActiveScreen     ; 
                        
                        jsr ClearGfxScreen          ; 
                        jsr ScreenStatusInitColor   ; 
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffGetReady ; 
                        ldx #<TabTextStatusGetReady ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
                        lda TabActualPlayerNumber   ; 
                        bne .GetPlayerTwo           ; 
                        
.GetPlayerOne           ldx #<TabTextStatusPlayerOne; 
                        jmp .OutPlayerNo            ; 
                        
.GetPlayerTwo           ldx #<TabTextStatusPlayerTwo; 
                        
.OutPlayerNo            ldy #BH_Scrn_Stat_OffPlayerNo ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset (Player two)
                        
                        lda #TabGameModeDemoToCave  ; move fleet to cave
                        ldx TabGameMode             ; 
                        bne .WaitSec_Ac             ; TabGameModeDemo or TabGameModeExit
                        
.WaitSec_05             lda #$05                    ; 
.WaitSec_Ac             sta TabWaitTimeTOD_Sec      ; 
                        jsr WaitTOD1_Sec            ; 
                        
ScreenStatusGetReadyX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusSave      Does    : Save status screen
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusSave        subroutine                  ; 
                        ldx #$00                    ; 
.GetNextByte            lda BH_Scrn_Stat + BH_Scrn_Stat_Row_01,x ; 
                        sta BH_Scrn_Stat_Save_Data,x ; 
                        
                        lda COLORAM + BH_Scrn_Stat_Row_01,x ; 
                        sta BH_Scrn_Stat_Save_Color,x ; 
                        
                        inx                         ; 
                        cpx #BH_Scrn_Stat_Rows * BH_Scrn_RowLen ; $c8
                        bne .GetNextByte            ; 
                        
ScreenStatusSaveX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusRestore   Does    : Restore status screen
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusRestore     subroutine                  ; 
                        ldx #$00                    ; 
.GetNextByte            lda BH_Scrn_Stat_Save_Data,x ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_Row_01,x ; 
                        
                        lda BH_Scrn_Stat_Save_Color,x ; 
                        sta COLORAM + BH_Scrn_Stat_Row_01,x ; 
                        
                        inx                         ; 
                        cpx #BH_Scrn_Stat_Rows * BH_Scrn_RowLen ; $c8
                        bne .GetNextByte            ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        inx                         ; 
                        txa                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffPlayer1 ; 
                        
                        lda #BH_StatColorPlayer     ; 
                        sta COLORAM + BH_Scrn_Stat_OffPlayer1 ; 
                        
ScreenStatusRestoreX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerValuesRestore   Does    : Sets game values for the actual player
;                       Expects : xr=player number
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerValuesRestore     subroutine                  ; 
                        lda TabPlayerLivesSav,x     ; 43
                        sta TabPlayerLives          ; 44
                        
                        lda TabPlayerCavePassSav,x  ; 45
                        sta TabPlayerCavePassed     ; 46
                        
                        lda TabPlayerTanksSav,x     ; 47
                        sta TabPlayerTanksLeft      ; 48
                        
                        lda TabPlayerShowScoresSav,x; 4d
                        sta TabPlayerShowScores     ; 4e
                        
                        lda TabPlayerScoreLoSav,x   ; 4b
                        sta TabPlayerScoreLo        ; 4c
                        
                        lda TabPlayerScoreHiSav,x   ; 4a overlap --> 49
                        sta TabPlayerScoreHi        ; 4b             4a
                        
                        lda TabPlayerHillTargsSav,x ; 50
                        sta TabPlayerHillTargsToHit ; 51
                        
                        lda TabBeachSpriteLevelSav,x; 52
                        sta TabBeachSpriteLevel     ; 53
                        
                        lda TabPlayerVesselActSav,x ; 54
                        sta TabPlayerVesselActive   ; 55
                        
                        lda TabPlayerNoUseSav2,x    ; 56
                        sta TabPlayerNoUse2         ; 57
                        
                        lda TabPlayerShipsSav,x     ; 48
                        sta TabPlayerShips          ; 49
                        
PlayerValuesRestoreX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerValuesSave      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerValuesSave        subroutine                  ; 
                        lda TabPlayerLives          ; 
                        sta TabPlayerLivesSav,x     ; 
                        
                        lda TabPlayerCavePassed     ; 
                        sta TabPlayerCavePassSav,x  ; 
                        
                        lda TabPlayerTanksLeft      ; 
                        sta TabPlayerTanksSav,x     ; 
                        
                        lda TabPlayerShowScores     ; 
                        sta TabPlayerShowScoresSav,x; 
                        
                        lda TabPlayerScoreLo        ; 
                        sta TabPlayerScoreLoSav,x   ; 
                        
                        lda TabPlayerScoreHi        ; 
                        sta TabPlayerScoreHiSav,x   ; 
                        
                        lda TabPlayerHillTargsToHit ; 
                        sta TabPlayerHillTargsSav,x ; 
                        
                        lda TabBeachSpriteLevel     ; 
                        sta TabBeachSpriteLevelSav,x; 
                        
                        lda TabPlayerVesselActive   ; 
                        sta TabPlayerVesselActSav,x ; 
                        
                        lda TabPlayerNoUse2         ; 
                        sta TabPlayerNoUseSav2,x    ; 
                        
                        lda TabPlayerShips          ; 
                        sta TabPlayerShipsSav,x     ; 
                        
PlayerValuesSaveX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerInit            Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerInit              subroutine                  ; 
                        ldx #$01                    ; 
.Players                lda #$0a                    ; 
                        sta TabPlayerLivesSav,x     ; 
                        
                        lda #$40                    ; 
                        sta TabMyFleetPosX          ; 
                        
                        lda #$01                    ; 
                        sta TabMyFleetMsbPosX       ; 
                        
                        lda #$32                    ; 
                        sta TabMyFleetPosY          ; 
                        
                        lda #$00                    ; 
                        sta TabJoysticAction        ; $00
                        sta SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        sta SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        jsr AcToTabSpritesActive    ; 
                        
                        lda TabGameMode             ; 
                        bne .InitAll                ; TabGameModeDemo or TabGameModeExit
                        
                        lda #$00                    ; 
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        
.InitAll                lda #$00                    ; 
                        sta TabPlayerScoreLoSav,x   ; 
                        sta TabPlayerScoreHiSav,x   ; 
                        sta TabPlayerShipsSav,x     ; 
                        sta TabPlayerCavePassSav,x  ; 
                        sta TabPlayerTanksSav,x     ; 
                        sta TabPlayerNoUseSav1,x    ; 
                        sta TabPlayerShowScoresSav,x; 
                        
                        lda #$0a                    ; 
                        sta TabPlayerHillTargsSav,x ; 
                        lda #$01                    ; 
                        sta TabP_ShowScores,x       ; 
                        lda #$3e                    ; ..#####. - one bit per ship
                        sta TabPlayerVesselActSav,x ; 
__BeachSpriteLevel      equ * + $01                 ; 
                        lda #$05                    ; 
                        sta TabBeachSpriteLevelSav,x; 
                        lda #$a8                    ; 
                        sta TabPlayerNoUseSav2,x    ; 
                        
                        dex                         ; 
                        bpl .Players                ; 
                        
                        ldx #$00                    ; player 1
                        stx TabActualPlayerNumber   ; 
                        
                        jsr PlayerValuesRestore     ; 
                        jsr PlayerSkillLevelInit    ; 
                        jsr ResetSid                ; 
                        
PlayerInitX             rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GameInit              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GameInit                subroutine                  ; 
                        ldx #$07                    ; 
.InitLoop               lda #$00                    ; 
                        sta TabVicScrollSoftPosX    ; 
                        sta TabBulletHight,x        ; 
                        sta TabPlaneDirReLi,x       ; TabPlaneDirReLiStop
                        sta TabPlaneDirUpDo,x       ; TabPlaneDirUpDo_None
                        sta TabPlaneFlies,x         ; 
                        sta TabMoveSpriteLeRiTime,x ; 
                        sta TabMoveSpriteUpDoTime,x ; 
                        sta T_9680,x                ; ???
                        sta TabMoveSpriteLeRi,x     ; TabMoveSprite_Ri
                        sta TabMoveSpriteUpDo,x     ; TabMoveSprite_Up
                        sta TabPlaneSpeedAppr,x     ; TabPlaneSpeedApprNone
                        sta TabSpriteStatus,x       ; TabSpriteStatusFree
                        sta TabPlaneDistance,x      ; 
                        sta GR_SpriteStepCount,x    ; 
                        sta TabWaitTankEneShoot     ; TabWaitTankEneShootNo
                        
                        lda #$01                    ; 
                        sta TabWaitTimes_Block01,x  ; IRQ dependent wait times
                        sta TabWaitTimes_Block03,x  ; 
                        sta TabWaitTimes_Block05,x  ; 
                        sta TabWaitFlags_Block01,x  ; 
                        sta TabWaitFlags_Block03,x  ; 
                        sta TabWaitFlags_Block05,x  ; 
                        sta TabWaitTimes_Block01Sav,x ; 
                        sta TabWaitTimes_Block03Sav,x ; 
                        sta TabWaitTimes_Block05Sav,x ; 
                        sta TabSeaVesselActive,x    ; TabSeaVesselActiveYes
                        sta TabUserAction           ; 
                        
                        dex                         ; 
                        bpl .InitLoop               ; 
                        
                        lda #$20                    ; 
                        sta TabPlaySfxVoc2_07       ; 
                        
GameInitX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusGameOver  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusGameOver    subroutine                  ; 
                        lda #$00                    ; 
                        jsr AcToTabSpritesActive    ; 
                        jsr ScreenStatusSave        ; 
                        jsr ScreenStatusInitColor   ; 
                        jsr ScreenStatusShowScore   ; 
                        
                        ldx #BH_Scrn_Stat_LenScore; 
.CpyScores              lda BH_Scrn_Stat + BH_Scrn_Stat_OffScoreGame,x; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffScoreOver,x; 
                        
                        lda #BLACK                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffScoreGame,x ; 
                        
                        lda #WHITE                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffScoreOver,x ; 
                        
                        dex                         ; 
                        bne .CpyScores              ; 
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffScoreTxtGO ; 
                        ldx #<TabTextStatusScore    ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset (Score)
                        
                        ldy #BH_Scrn_Stat_OffTextGO ; 
                        ldx #<TabTextStatusGameOver ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
                        lda TabActualPlayerNumber   ; 
                        bne .GetTxtPlayer_2         ; 
                        
                        ldx #<TabTextStatusPlayerOne; 
                        jmp .OutTxtPlayer_2         ; 
                        
.GetTxtPlayer_2         ldx #<TabTextStatusPlayerTwo; 
                        
.OutTxtPlayer_2         ldy #BH_Scrn_Stat_OffPlayerNo ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
                        lda #$05                    ; 
                        sta TabWaitTimeTOD_Sec      ; 
                        jsr WaitTOD1_Sec            ; 
                        jsr ScreenStatusRestore     ; 
ScreenStatusGameOverX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CheckGameIdleTime     Does    : Check if in game (reset TOD2 then) or demo mode
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CheckGameIdleTime       subroutine                  ; 
                        lda TabGameMode             ; 
                        beq .ChkUserAction          ; TabGameModeGame   
                        
                        rts                         ; TabGameModeDemo
                        
.ChkUserAction          lda TabUserAction           ; 
                        beq .ChkIdleTime            ; TabUserActionNo   
                        
.SetUserAction          lda #TabUserActionNo        ; 
                        sta TabUserAction           ; 
                        
                        jsr ResetTOD2               ; reset idle time
                        
.Exit                   rts                         ; 
                        
.ChkIdleTime            lda TO2MIN                  ; CIA2($DD0A) Time of Day Clock Minutes
                        cmp #$01                    ; up to 1 minute idle time is tolerable
                        bne CheckGameIdleTimeX      ; 
                        
.IdleTimeUp             lda #TabGameModeExit        ; idle too long - exit game
                        sta TabGameMode             ; 
                        
CheckGameIdleTimeX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ResetTOD2             Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ResetTOD2               subroutine                  ; 
                        lda #$00                    ; 
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        sta TO2MIN                  ; CIA2($DD0A) Time of Day Clock Minutes
                        sta TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        sta TO2TEN                  ; CIA2($DD08) Time of Day Clock Tenths
                        
ResetTOD2X              rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusTextOut   Does    : 
;                       Expects : xr=input text offset  yr=output text offset
;                                 xr=length for blank out if TabScrnStatusTextColor=$ff
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusTextOut     subroutine                  ; 
ScreenStatusTextOutRow1 lda #<BH_Scrn_Stat_Row_01   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow2 lda #<BH_Scrn_Stat_Row_02   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow3 lda #<BH_Scrn_Stat_Row_03   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow4 lda #<BH_Scrn_Stat_Row_04   ; 
                        jmp .SetPtrLo               ; 
ScreenStatusTextOutRow5 lda #<BH_Scrn_Stat_Row_05   ; 
                        
.SetPtrLo               sta BLZ_ScrnRowPtrStatusOutLo ; 
                        sta BLZ_ScrnRowPtrColoRamOutLo ; 
                        
                        lda #>[BH_Scrn_Stat + BH_Scrn_Stat_Row_01] ; 
                        sta BLZ_ScrnRowPtrStatusOutHi ; 
                        
                        lda #>[COLORAM        + BH_Scrn_Stat_Row_01] ; 
                        sta BLZ_ScrnRowPtrColoRamOutHi ; 
                        
                        lda TabScrnStatusTextColor  ; 
                        bmi .GetBlank               ; flag: $ff - blanks out
                        
.GetText                lda TabTextStatus,x         ; 
                        beq ScreenStatusTextOutX    ; end of text
                        sta (BLZ_ScrnRowPtrStatusOut),y ; 
                        
                        lda TabScrnStatusTextColor  ; 
                        sta (BLZ_ScrnRowPtrColoRamOut),y ; 
                        
                        inx                         ; 
                        iny                         ; 
                        jmp .GetText                ; 
                        
.GetBlank               lda #$20                    ; 
.PutBlank               sta (BLZ_ScrnRowPtrStatusOut),y ; 
                        
                        iny                         ; 
                        dex                         ; 
                        bne .PutBlank               ; 
                        
ScreenStatusTextOutX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetSprtSprtCollReg    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetSprtSprtCollReg      subroutine                  ; 
                        lda SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        sta TabSprtSprtCollReg      ; 
                        
GetSprtSprtCollRegX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; GetSprtBackCollReg    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
GetSprtBackCollReg      lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        sta TabSprtBackCollReg      ; 
                        
GetSprtBackCollRegX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerAmmoDec         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerAmmoDec           subroutine                  ; 
                        lda TabPlayerAmmo           ; 
                        bne .DecAmmo                ; 
                        
                        rts                         ; 
                        
.DecAmmo                dec TabPlayerAmmo           ; 
                        
PlayerAmmoDecX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlayerAmmoInc         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlayerAmmoInc           subroutine                  ; 
                        lda TabWaitFlagAmmoInc      ; 
                        beq .IncAmmo                ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.IncAmmo                inc TabWaitFlagAmmoInc      ; TabWaitFlag_Yes
                        
                        lda TabPlayerAmmo           ; 
__SkillAmmoInc          equ * + $01                 ; 
                        cmp #$0a                    ; 
                        bcs PlayerAmmoIncX          ; greater/equal
                        
                        inc TabPlayerAmmo           ; 
                        
PlayerAmmoIncX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusAmmoDec   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusAmmoFlack   subroutine                  ; 
                        lda #BH_StatColorAmmoFlack  ; 
                        bne ScreenStatusAmmoDec     ; always
                        
ScreenStatusAmmoShip    subroutine                  ; 
                        lda #BH_StatColorAmmoShip   ; 
                        bne ScreenStatusAmmoDec     ; always
                        
ScreenStatusAmmoTank    subroutine                  ; 
                        lda #BH_StatColorAmmoTank   ; 
                        
ScreenStatusAmmoDec     subroutine                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldx #$00                    ; 
                        ldy TabPlayerAmmo           ; 
                        dey                         ; 
                        bmi .BlankOut               ; 
                        
.BoxesOut               lda #$7c                    ; display ammo boxes
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffAmmoFlackCube,x ; 
                        
                        lda TabScrnStatusTextColor  ; 
                        sta COLORAM + BH_Scrn_Stat_OffAmmoFlackCube,x ; 
                        
                        inx                         ; 
                        dey                         ; 
                        bpl .BoxesOut               ; 
                        
.BlankOut               cpx #BH_Scrn_Stat_LenAmmoFlackCube ; 
                        bcs ScreenStatusAmmoDecX    ; 
                        
                        lda #$20                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffAmmoFlackCube,x ; 
                        inx                         ; 
                        jmp .BlankOut               ; 
                        
ScreenStatusAmmoDecX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearDirInJoyAction   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearDirInJoyAction     subroutine                  ; 
                        eor #$ff                    ; flip selected bits
                        and TabJoysticAction        ; clear
                        sta TabJoysticAction        ; 
                        
ClearDirInJoyActionX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowShips Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowShips   subroutine                  ; 
                        ldx #$00                    ; 
                        ldy TabPlayerLives          ; 
                        beq .ChkMax                 ; 
                        
.GetIndicator           lda #$51                    ; "O"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffShipsO,x ; indicators
                        
                        lda #BH_StatColorShipsCavePass ; preset
                        
                        cpy TabPlayerShips          ; 
                        bcc .SetColor               ; lower
                        beq .SetColor               ; equal
                        
                        lda #BH_StatColorShipsGame  ; greater
                        
.SetColor               sta COLORAM + BH_Scrn_Stat_OffShipsO,x ; 
                        inx                         ; 
                        dey                         ; 
                        bne .GetIndicator           ; 
                        
.ChkMax                 cpx #BH_ShipsMax            ; 
                        bne .BlankOut               ; 
                        
.Exit                   rts                         ; 
                        
.BlankOut               lda #$20                    ; " "
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffShipsO,x ; 
                        
                        inx                         ; 
ScreenStatusShowShipsX  jmp .ChkMax                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
                        seg HICODE
; -------------------------------------------------------------------------------------------------------------- ;
                        * equ $6580                 ; Beach Head - HICODE.prg
; -------------------------------------------------------------------------------------------------------------- ;
; ShowStartScreen       Does    : Called from BASIC
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShowStartScreen         jsr GameInit                ; .hbu001. - BASIC 210
                        
                        ldx #$00                    ; 
.CopyKernalToRam        lda Kernel_Vectors,x        ; 
                        sta Kernel_Vectors,x        ; 
                        dex                         ; 
                        bne .CopyKernalToRam        ; 
                        
                        jsr C64IrqScrollInit        ; 
                        
                        lda #B_Koff                 ; 
                        sta R6510                   ; 
                        
                        jsr ScreenStatusInitColor   ; 
                        
                        lda #YELLOW                 ; 
                        sta TabBackGroundColorSav   ; 
                        
                        lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; 
                        
                        lda #$01                    ; screen A
                        sta TabTankActiveScreen     ; 
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        ora #$08                    ; ....#... - 40 columns
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        ldx #<ScreenDataBayShore    ; 
                        ldy #>ScreenDataBayShore    ; 
                        jsr ScreenGraphicCharSet    ; 
                        
                        ldx #<ScreenDataIntro       ; 
                        ldy #>ScreenDataIntro       ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Intro
                        
                        ldy #BH_Scrn_Stat_OffInfoBruce ; 
                        ldx #<TabTextStatusByBruce  ; by bruce carver
                        lda #BLACK                  ; 
                        sta TabScrnStatusTextColor  ; 
                        jsr ScreenStatusTextOutRow1 ; xr=input text offset  yr=output text offset
                        
                        ldy #BH_Scrn_Stat_OffInfoCR ; 
                        ldx #<TabTextStatusCopyright; (c) 1983  access software inc
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
                        lda #$05                    ; 
                        sta TabWaitTimeTOD_Sec      ; 
                        
                        jsr WaitTOD1_Sec            ; 
                        
                        lda #$00                    ; 
                        sta TabBackGroundColorSav   ; BLACK
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TO2HRS                  ; CIA2($DD0B) Time of Day Clock Hours
                        
                        lda #BIKon                  ; 
                        sta R6510                   ; 
                        
                        jsr C64Reset                ; 
ShowStartScreenX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_BayAreaMap    Does    : Display the bay area map - Main Loop
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_BayAreaMap      lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; 
                        
                        lda #$01                    ; screen A
                        sta TabTankActiveScreen     ; 
                        
                        lda SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        ora #$08                    ; ....#... - 40 columns
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        jsr InitBayAreaMap          ; 
                        
                        lda #$03                    ; ......##
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #TabBayMapSeaChaptersYes; bay shootouts first
                        sta TabBayMapSeaChapters    ; 
                        
.BayAreaMap_Loop        jsr MapMyFleetGetMoves      ; get joystick directions
                        jsr MapMyFleetSetMoves      ; try to move fleet
                        jsr MapChkMyFleetEnterCave  ; to --> Chapter 1  : Cave Crossing
                        jsr MapCloseBayAndChase     ; 
                        jsr MapHitMyFleetShootOut   ; to --> Chapter 2+3: Sea  Shootouts
                        jsr MapHitBeachTankHill     ; to --> Chapter 4+5: Tank Shootouts
                        jsr MapBlinkTargetsPing     ; 
                        jsr CheckGamePause          ; with C= commodore key
                        jsr ScreenStatusShowScore   ; 
                        jsr MapChkFinished          ; idle time/death - 2 level return to BASIC via GameStart
                        jsr ScreenStatusShowShips   ; 
                        jsr CheckGameIdleTime       ; 
                        jsr MapDemo                 ; 
                        
Chapter_BayAreaMapX     jmp .BayAreaMap_Loop        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitBayAreaMap        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitBayAreaMap          subroutine                  ; 
                        jsr ScreenStatusGetReady    ; display 'get ready' screen
                        
                        ldx #<ScreenDataBayMap      ; 
                        ldy #>ScreenDataBayMap      ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Map
                        jsr ScreenStatusMap         ; 
                        jsr SpritesReset            ; 
                        
                        ldx #BH_ChapterBayAreaMap   ; 
                        jsr SpriteDataCopyInit      ; 
                        
                        lda #$01                    ; .......#
                        sta TabSfxToPlay            ; 
                        
                        jsr ResetSid                ; 
                        lda #$50                    ; 
                        sta SUREL2                  ; SID($D40D) Oscillator 2 Sustain/Release
                        lda #$81                    ; 
                        sta VCREG2                  ; SID($D40B) Oscillator 2 Control
                        
                        lda #$06                    ; 
                        sta TabWaitTimeBayLeRiSav   ; my fleet
                        sta TabWaitTimeBayUpDoSav   ; 
                        
                        lda #$05                    ; 
                        sta TabWaitTimeBayLeRiSavE  ; enemy fleet - moves a bit quicker than my fleet
                        sta TabWaitTimeBayUpDoSavE  ; 
                        
                        lda #$20                    ; 
                        sta TabWaitTimeBayBlinkSav  ; 
                        
                        lda #$00                    ; 
                        sta TabMoveSpriteLeRiTime   ; 
                        sta TabMoveSpriteUpDoTime   ; 
                        
                        lda #BH_BayFleetMyStartPosX ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #BH_BayFleetMyStartPosX_Msb ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        lda #BH_BayFleetMyStartPosY ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Map_03          ; fleet 4 ships
                        sta BH_SpritePointer00_A    ; 
                        
                        lda #BH_BayFleetEnemyStartPosX ; 
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #BH_BayFleetEnemyStartPosY ; 
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Map_08          ; fleet 5 ships
                        sta BH_SpritePointer01_A    ; 
                        
                        lda #BH_SprtColorFleetMy    ; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #BH_SprtColorFleetEnemy ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        
                        lda #$00                    ; 
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
InitBayAreaMapX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetGetMoves    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetGetMoves      subroutine                  ; 
                        lda TabWaitFlagMoveUpDo     ; TabWaitFlag_Yes
                        bne .StickReset             ; 
                        
.PrepMyFleetMove        inc TabWaitFlagMoveUpDo     ; TabWaitFlag_Yes
                        
.ChkLeft                lda #CIA_Joy_Le             ; .....#..
                        bit TabJoysticAction        ; 
                        beq .ChkRight               ; 
                        
.WasLeft                jsr MapMyFleetMoveSetLeft   ; 
                        jmp .ChkUp                  ; 
                        
.ChkRight               lda #CIA_Joy_Ri             ; ....#...
                        bit TabJoysticAction        ; 
                        beq .ChkUp                  ; 
                        
.WasRight               jsr MapMyFleetMoveSetRight  ; 
                        
.ChkUp                  lda #CIA_Joy_Up             ; .......#
                        bit TabJoysticAction        ; 
                        beq .ChkDown                ; 
                        
.WasUp                  jsr MapMyFleetMoveSetUp     ; 
                        jmp .StickReset             ; 
                        
.ChkDown                lda #CIA_Joy_Do             ; ......#.
                        bit TabJoysticAction        ; 
                        beq .StickReset             ; 
                        
.WasDown                jsr MapMyFleetMoveSetDown   ; 
                        
.StickReset             lda #$00                    ; 
                        sta TabJoysticAction        ; 
                        
                        lda TabWaitFlagMoveLeRi     ; 
                        beq .MoveMyFleet            ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.MoveMyFleet            inc TabWaitFlagMoveLeRi     ; TabWaitFlag_Yes
                        
                        lda TabMoveSpriteLeRiTime   ; 
.ChkMoveLeft            cmp #$ff                    ; 
                        bne .ChkMoveRight           ; 
                        
                        jsr MapMyFleetMoveLeft      ; 
                        jmp .ChkMoveDown            ; 
                        
.ChkMoveRight           cmp #$01                    ; 
                        bne .ChkMoveDown            ; 
                        
                        jsr MapMyFleetMoveRight     ; 
                        
.ChkMoveDown            lda TabMoveSpriteUpDoTime   ; 
                        cmp #$ff                    ; 
                        bne .ChkMoveUp              ; 
                        
                        jsr MapMyFleetMoveDown      ; 
                        jmp MapMyFleetGetMovesX     ; 
                        
.ChkMoveUp              cmp #$01                    ; 
                        bne MapMyFleetGetMovesX     ; 
                        
                        jsr MapMyFleetMoveUp        ; 
                        
MapMyFleetGetMovesX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveUp      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveUp        subroutine                  ; 
                        lda #$2e                    ; 
                        cmp SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        beq MapMyFleetMoveUpX       ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta TabMyFleetPosY          ; 
                        
                        dec SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
MapMyFleetMoveUpX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveDown    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveDown      subroutine                  ; 
                        lda #$b9                    ; 
                        cmp SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        beq MapMyFleetMoveDownX     ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta TabMyFleetPosY          ; 
                        
                        inc SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
MapMyFleetMoveDownX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveLeft    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveLeft      subroutine                  ; 
                        lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .GetPosX                ; 
                        
                        lda #$14                    ; 
                        cmp SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .Exit                   ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabMyFleetPosX          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        dec SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
.Exit                   rts                         ; 
                        
.GetPosX                lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabMyFleetPosX          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        dec SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bpl .Exit                   ; 
                        
                        lda #$fe                    ; #######.
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
MapMyFleetMoveLeftX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveRight   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveRight     subroutine                  ; 
                        lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .GetPosX                ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabMyFleetPosX          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        inc SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bne .Exit                   ; 
                        
                        lda #$01                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.Exit                   rts                         ; 
                        
.GetPosX                lda #$44                    ; 
                        cmp SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .Exit                   ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabMyFleetPosX          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        inc SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
MapMyFleetMoveRightX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetSetMoves    Does  : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetSetMoves      subroutine                  ; 
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        and #$01                    ; .......#
                        bne .GetFleetPosX           ; 
                        
                        rts                         ; hit an obstacle so don't move
                        
.GetFleetPosX           lda TabMyFleetPosX          ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda TabMyFleetMsbPosX       ; 
                        and #$01                    ; 
                        beq .GetFleetMsbPosX        ; 
                        
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .SetFleetMsbPosX        ; 
                        
.GetFleetMsbPosX        lda #$fe                    ; 
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.SetFleetMsbPosX        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda TabMyFleetPosY          ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
MapMyFleetSetMovesX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapCloseBayAndChase   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapCloseBayAndChase     subroutine                  ; 
                        lda TabBayMapSeaChapters    ; 
                        bne .ChkCavePassed          ; TabBayMapSeaChaptersYes
                        
.Exit                   rts                         ; 
                        
.ChkCavePassed          lda TabPlayerCavePassed     ; 
                        bne .ChkCavePassedGame      ; 
                        
                        lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .GetMyFleetPos          ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #BH_BayFleetChaseGoPosY ; 
                        beq .SetBayEnterDirect      ; 
                        
                        rts                         ; 
                        
.GetMyFleetPos          lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        ldx SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cpx #BH_BayPassageLeftPosX  ; 
                        bcc .ChkAdjustPosY          ; lower
                        
                        sec                         ; 
                        sbc #BH_BayRightAdjustPosY  ; posY adjustment
                        
.ChkAdjustPosY          cmp #BH_BayFleetChaseGoPosY - BH_BayRightAdjustPosY - $01 ; 
                        bcc .Exit                   ; lower
                        
.SetBayEnterDirect      lda #TabCavePassedNo        ; 
                        sta TabPlayerCavePassed     ; 
                        
.CloseBay               lda #$52                    ; after passing the entrance/leaving cave
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayTop + $00 ; $49d2
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayMid + $00 ; $49fc
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayBot + $00 ; $4a26
                        
                        lda #$53                    ; 
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayTop + $01 ; $49d3
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayMid + $01 ; $49fd
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffCloseBayBot + $01 ; $4a27
                        jmp .ChkMoveFleetEnemy      ; 
                        
.ChkCavePassedGame      cmp #TabCavePassedYesGame   ; 
                        bne .ChkMoveFleetEnemy      ; 
                        
.ChkChasePosX           lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #BH_BayFleetChaseGoPosX ; 
                        bcc .ChkChasePosY           ; 
                        
.SetBayEnterViaCave     lda #TabCavePassedYesChase  ; 
                        sta TabPlayerCavePassed     ; 
                        jmp .CloseBay               ; 
                        
.ChkChasePosY           lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #BH_BayFleetChaseGoPosY ; 
                        bne .Exit                   ; 
                        
                        jmp .SetBayEnterViaCave     ; 
                        
.ChkMoveFleetEnemy      lda TabWaitFlagMoveEnemy    ; 
                        beq .MoveFleetEnemy         ; 
                        
                        rts                         ; 
                        
.MoveFleetEnemy         lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveEnemy    ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .ChkFleetsHitPosY       ; 
                        bcs .IncFleetEnemyPosX      ; greater
                        
                        lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .IncFleetEnemyPosX      ; 
                        
.DecFleetEnemyPosX      lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$01                    ; 
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .ChkFleetsHitPosY       ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$fd                    ; ######.#
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .ChkFleetsHitPosY       ; 
                        
.IncFleetEnemyPosX      inc SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bne .ChkFleetsHitPosY       ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        ora #$02                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.ChkFleetsHitPosY       lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        beq MapCloseBayAndChaseX    ; 
                        bcs .IncFleetEnemyPosY      ; 
                        
.DecFleetEnemyPosY      dec SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        rts                         ; 
                        
.IncFleetEnemyPosY      inc SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
MapCloseBayAndChaseX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
InitBayArea             subroutine                  ; 
                        lda #TabBayMapSeaChaptersNo ; 
                        sta TabBayMapSeaChapters    ; 
                        jsr InitBayAreaMap          ; 
                        
.SetMyFleet             lda TabMyFleetPosX          ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda TabMyFleetMsbPosX       ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        lda TabMyFleetPosY          ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$00                    ; 
.Clear                  jsr AcToTabSpritesActive    ; 
                        
                        lda #$01                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
InitBayAreaX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapDemo               Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapDemo                 subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .ChkModeMove2Cave       ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.ChkModeMove2Cave       cmp #TabGameModeDemoToCave  ; move fleet to cave
                        bne .ChkModeIsInBay         ; 
                        
.MovFleet2Cave          lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$05                    ; 
                        beq .IniResetTOD2_Cave      ; move myFleet for 5secs only
                        
                        lda #CIA_Joy_Le             ; move fleet left
                        sta TabJoysticAction        ; 
                        
                        rts                         ; 
                        
.IniResetTOD2_Cave      jsr ResetTOD2               ; 
                        
.SetMyFleet2Cave        lda #$fe                    ; 
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #BH_BayFleetHitCavePosX ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #BH_BayFleetHitCavePosY ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        rts                         ; 
                        
.ChkModeIsInBay         cmp #TabGameModeDemoInBay   ; fleet encounters enemy fleet
                        bne .MovMyFleet2Beach       ; TabGameModeDemoToBeach
                        
.SetMyFleetCavePassed   lda #BH_BayFleetChaseGoPosX ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #BH_BayFleetChaseGoPosY ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        rts                         ; 
                        
.MovMyFleet2Beach       lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$03                    ; move myFleet for 3secs only
                        beq .SetMyFleet2Beach       ; 
                        
                        lda #[CIA_Joy_Do + CIA_Joy_Le] ; move fleet down left
                        sta TabJoysticAction        ; 
                        
                        rts                         ; 
                        
.SetMyFleet2Beach       jsr ResetTOD2               ; 
                        
                        lda #BH_BayFleetHitBeachPosX; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #BH_BayFleetHitBeachPosY; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
MapDemoX                rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusMap       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusMap         subroutine                  ; 
                        jsr ScreenStatusInitColor   ; 
                        jsr ScreenStatusInitGfx     ; 
                        jsr ScreenStatusInitMap     ; 
                        
                        lda #BH_StatColorAmmoFlackTxt ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoFlackTxt ; 
                        ldx #<TabTextStatus40mm     ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorAmmoShipTxt; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoShipTxt ;
                        ldx #<TabTextStatus250mm    ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorAmmoFlack  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmo   ; 
                        ldx #<TabTextStatusAmmoCube ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorAmmoShip   ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmo   ; 
                        ldx #<TabTextStatusAmmoCube ; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorArialRecTxt; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapAerial ; 
                        ldx #<TabTextStatusArialRec ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
ScreenStatusMapX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapChkMyFleetEnterCave Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapChkMyFleetEnterCave  subroutine                  ; 
                        lda TabPlayerCavePassed     ; 
                        beq .ChkCaveMapPos          ; 
                        
.NoCaveEntrance         rts                         ; 
                        
.ChkCaveMapPos          lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$01                    ; 
                        bne .NoCaveEntrance         ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$80                    ; 
                        bcc .NoCaveEntrance         ; 
                        
                        cmp #$88                    ; 
                        bcs .NoCaveEntrance         ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$4a                    ; 
                        bcc .NoCaveEntrance         ; 
                        
                        cmp #$5a                    ; 
                        bcs .NoCaveEntrance         ; 
                        
                        jsr ChapterStartBell        ; 
MapChkMyFleetEnterCaveX jmp Chapter_CaveCrossings   ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_CaveCrossings Does    : Chapter: Mine and torpedo cave
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_CaveCrossings   subroutine                  ; 
                        jsr ScreenStatusSave        ; 
                        jsr ScreenStatusGetReady    ; 
                        jsr ScreenStatusRestore     ; 
                        
                        ldx #<ScreenDataCave        ; 
                        ldy #>ScreenDataCave        ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Cave
                        
                        lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapAerial ; 
                        ldx #BH_Scrn_Stat_LenMapAerial ; clear 'AERIAL RECON'
                        jsr ScreenStatusTextOutRow5 ; xr=lenght for blank out  yr=output text offset
                        
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        ldx #BH_ChapterCaveCrossings; 
                        jsr SpriteDataCopyInit      ; 
                        jsr CaveInitTorpedoes       ; 
                        jsr CaveSfxTorpedoShot      ; 
                        jsr CaveInitShip            ; 
                        
.CaveCrossings_Loop     jsr CaveMoveShip            ; 
                        jsr GetSprtSprtCollReg      ; sprite/sprite collision
                        jsr CaveChkHits             ; by torpedoes or background
                        jsr CaveChkExitShip         ; 
                        jsr CaveChkExitTorpedo      ; 
                        jsr ScreenStatusShowShips   ; 
                        jsr CaveSfxShipEngine       ; 
                        jsr CheckGamePause          ; on C= commodore key
                        jsr ScreenStatusShowScore   ; 
                        jsr CheckGameFinished       ; idle time/death - 3 level return to BASIC via GameStart
                        jsr CheckGameIdleTime       ; 
                        jsr CaveDemo                ; 
                        
Chapter_CaveCrossingsX  jmp .CaveCrossings_Loop     ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveInitTorpedoes     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveInitTorpedoes       subroutine                  ; 
                        lda #SP_Ptr_Cave_03         ; torpedo nee
                        sta BH_SpritePointer01_A    ; 
                        lda #SP_Ptr_Cave_03         ; torpedo nee
                        sta BH_SpritePointer02_A    ; 
                        lda #SP_Ptr_Cave_03         ; torpedo nee
                        sta BH_SpritePointer03_A    ; 
                        lda #SP_Ptr_Cave_04         ; torpedo e
                        sta BH_SpritePointer04_A    ; 
                        lda #SP_Ptr_Cave_04        ; torpedo e
                        sta BH_SpritePointer05_A    ; 
                        lda #SP_Ptr_Cave_04         ; torpedo e
                        sta BH_SpritePointer06_A    ; 
                        lda #SP_Ptr_Cave_04         ; torpedo e
                        sta BH_SpritePointer07_A    ; 
                        
                        lda #BH_SprtColorCaveTorp_1 ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        sta SP6COL                  ; VIC($D02D) Color Sprite 6
                        
                        lda #BH_SprtColorCaveTorp_2 ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        sta SP5COL                  ; VIC($D02C) Color Sprite 5
                        
                        lda #BH_SprtColorCaveTorp_3 ; 
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        sta SP7COL                  ; VIC($D02E) Color Sprite 7
                        
                        lda #$00                    ; 
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$fe                    ; #######.
                        sta SPBGPR                  ; VIC($D01B) Sprite to Foreground Display Priority
                        
__CaveSkillTorpedo_01   equ * + $01                 ;                         
                        lda #$9e                    ; 
                        sta TabMoveTorpedo_01       ; 
__CaveSkillTorpedo_02   equ * + $01                 ;                         
                        lda #$8f                    ; 
                        sta TabMoveTorpedo_02       ; 
__CaveSkillTorpedo_03   equ * + $01                 ;                         
                        lda #$18                    ; 
                        sta TabMoveTorpedo_03       ; 
__CaveSkillTorpedo_04   equ * + $01                 ;                         
                        lda #$0c                    ; 
                        sta TabMoveTorpedo_04       ; 
__CaveSkillTorpedo_05   equ * + $01                 ;                         
                        lda #$98                    ; 
                        sta TabMoveTorpedo_05       ; 
__CaveSkillTorpedo_06   equ * + $01                 ;                         
                        lda #$8c                    ; 
                        sta TabMoveTorpedo_06       ; 
__CaveSkillTorpedo_07   equ * + $01                 ;                         
                        lda #$1e                    ; 
                        sta TabMoveTorpedo_07       ; 
__CaveSkillTorpedo_08   equ * + $01                 ;                         
                        lda #$14                    ; 
                        sta TabMoveTorpedo_08       ; 
__CaveSkillTorpedo_09   equ * + $01                 ;                         
                        lda #$a0                    ; 
                        sta TabMoveTorpedo_09       ; 
__CaveSkillTorpedo_10   equ * + $01                 ;                         
                        lda #$1e                    ; 
                        sta TabMoveTorpedo_0a       ; 
                        
                        lda #$00                    ; 
                        sta TabMoveTorpedo_Val_1    ; 
                        sta TabMoveTorpedo_Val_2    ; 
                        sta TabMoveTorpedo_Val_3    ; 
                        sta TabMoveTorpedo_Val_4    ; 
                        
                        lda #$05                    ; 
                        sta TabWaitTimeTOD_10th     ; 
                        
                        ldx #$07                    ; 
.IniTorpedoPos          jsr InitTorpedoSpritePos    ; 
                        
                        lda #$01                    ; 
                        sta TabSpriteActiveCave,x   ; 
                        
                        dex                         ; 
                        bne .IniTorpedoPos          ; 
                        
                        lda #$fe                    ; #######.
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
CaveInitTorpedoesX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; InitTorpedoSpritePos  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
InitTorpedoSpritePos    subroutine                  ; 
.ChkSprite_01           cpx #$01                    ; 
                        bne .ChkSprite_02           ; 
                        
                        lda #$df                    ; 
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$27                    ; 
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        lda #$fd                    ; ######.#
                        
.ClrMsbX                and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        rts                         ; 
                        
.ChkSprite_02           cpx #$02                    ; 
                        bne .ChkSprite_03           ; 
                        
.SetSpritePos_02        lda #$26                    ; 
                        sta SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$b7                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        lda #$fb                    ; #####.##
                        jmp .ClrMsbX                ; 
                        
.ChkSprite_03           cpx #$03                    ; 
                        bne .ChkSprite_04           ; 
                        
.SetSpritePos_03        lda #$3f                    ; 
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$5f                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        lda #$08                    ; ....#...
                        jmp .SetMsbX                ; 
                        
.ChkSprite_04           cpx #$04                    ; 
                        bne .ChkSprite_05           ; 
                        
.SetSpritePos_04        lda #$0c                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$50                    ; 
                        bcs .SetSpritePosY_04       ; 
                        
                        lda #$a1                    ; 
                        
.SetSpritePosY_04       sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        lda #$ef                    ; ###.####
                        jmp .ClrMsbX                ; 
                        
.SetMsbX                ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        rts                         ; 
                        
.ChkSprite_05           cpx #$05                    ; 
                        bne .ChkSprite_06           ; 
                        
.SetSpritePos_05        lda #$0c                    ; 
                        sta SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$7f                    ; 
                        sta SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        lda #$df                    ; ##.#####
                        jmp .ClrMsbX                ; 
                        
.ChkSprite_06           cpx #$06                    ; 
                        bne .SetSpritePos_07        ; 
                        
.SetSpritePos_06        lda #$5f                    ; 
                        sta SP6X                    ; VIC($D00C) Sprite 6 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$5f                    ; 
                        sta SP6Y                    ; VIC($D00D) Sprite 6 Y-Pos (Bits 0-7)
                        lda #$40                    ; .#......
                        jmp .SetMsbX                ; 
                        
.SetSpritePos_07        lda #$0c                    ; 
                        sta SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$43                    ; 
                        sta SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        lda #$7f                    ; .#######
InitTorpedoSpritePosX   jmp .ClrMsbX                ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveInitShip          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveInitShip            subroutine                  ; 
                        lda #SP_Ptr_Cave_09         ; ship nww
                        sta BH_SpritePointer00_A    ; 
                        
                        jsr ResetVoc3               ; 
                        lda #$00                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$00                    ; 
                        sta PWLO3                   ; SID($D410) Oscillator 3 Pulse Waveform Width (low byte)
                        lda #$08                    ; 
                        sta PWHI3                   ; SID($D411) Oscillator 3 Pulse Waveform Width (high nybble)
                        lda #$80                    ; 
                        sta SUREL3                  ; SID($D414) Oscillator 3 Sustain/Release
                        lda #$41                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
                        lda #TabCaveShipDirInit     ; 
                        sta TabCaveShipDir          ; 
                        
                        lda #$10                    ; 
                        sta TabPlaySfxVoc2_00       ; 
                        sta TabPlaySfxVoc1_00       ; 
                        
                        lda #$08                    ; 
                        sta TabPlaySfxVoc2_01       ; 
                        sta TabPlaySfxVoc1_01       ; 
                        
                        lda #TabCaveShipSpeedInit   ; 
                        sta TabCaveShipSpeed        ; 
                        
                        lda #BH_SprtColorShip       ; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #$20                    ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$01                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$a2                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$01                    ; 
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #TabCaveShipExitsNo     ; 
                        sta TabCaveShipExits        ; 
                        jsr GetSprtSprtCollReg      ; sprite/sprite collision
                        
                        lda #$00                    ; 
                        sta TabSprtSprtCollReg      ; 
                        
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        
CaveInitShipX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveMoveShip          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveMoveShip            subroutine                  ; 
                        lda TabWaitFlagCaveShipLeRi ; 
                        bne .ClrJoyLeRi             ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagCaveShipLeRi ; TabWaitFlag_Yes
                        
.ChkLeft                lda #CIA_Joy_Le             ; .....#..
                        bit TabJoysticAction        ; 
                        beq .ChkRight               ; 
                        
.WasLeft                jsr ClearDirInJoyAction     ; 
                        
                        lda TabCaveShipDir          ; 
                        beq .ClrJoyLeRi             ; TabCaveShipDirMin
                        
                        dec TabCaveShipDir          ; 
                        
                        jmp .ClrJoyLeRi             ; 
                        
.ChkRight               lda #CIA_Joy_Ri             ; ....#...
                        bit TabJoysticAction        ; 
                        beq .ClrJoyLeRi             ; 
                        
.WasRight               jsr ClearDirInJoyAction     ; 
                        
                        lda TabCaveShipDir          ; 
                        cmp #TabCaveShipDirMax      ; 
                        beq .ClrJoyLeRi             ; 
                        
                        inc TabCaveShipDir          ; 
                        
.ClrJoyLeRi             lda #[CIA_Joy_Le + CIA_Joy_Ri] ; ....##..
                        jsr ClearDirInJoyAction     ; 
                        
                        lda TabCaveShipDir          ; 
                        clc                         ; 
                        adc #SP_Ptr_Cave_08         ; ship w
                        sta BH_SpritePointer00_A    ; 
                        
                        lda TabWaitFlagCaveShipUpDo ; 
                        bne .SetShipMoves           ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagCaveShipUpDo ; TabWaitFlag_Yes
                        
.ChkUp                  lda #CIA_Joy_Up             ; .......#
                        bit TabJoysticAction        ; 
                        beq .ChkDown                ; 
                        
                        jsr ClearDirInJoyAction     ; 
                        
                        lda TabCaveShipSpeed        ; 
                        cmp #TabCaveShipSpeedMax    ; 
                        beq .SetShipMoves           ; 
                        
                        inc TabCaveShipSpeed        ; 
                        
                        jmp .SetShipMoves           ; 
                        
.ChkDown                lda #CIA_Joy_Do             ; ......#.
                        bit TabJoysticAction        ; 
                        beq .SetShipMoves           ; 
                        
                        jsr ClearDirInJoyAction     ; 
                        
                        lda TabCaveShipSpeed        ; 
                        beq .SetShipMoves           ; 
                        
                        dec TabCaveShipSpeed        ; 
                        
.SetShipMoves           lda #TabCaveShipDirMax      ; 
                        sec                         ; 
                        sbc TabCaveShipDir          ; 
                        jsr CaveGetMoveValue        ; 
                        
                        ora #TabMoveSprite_Le       ; #....... - left
                        sta TabMoveSpriteLeRi       ; 
                        
                        lda TabCaveShipDir          ; 
                        jsr CaveGetMoveValue        ; 
                        
                        sta TabMoveSpriteUpDo       ; 
                        
                        lda #$01                    ; 
                        sta TabSpriteActiveCave     ; 
                        
CaveMoveShipX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveGetMoveValue     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveGetMoveValue        subroutine                  ; 
                        sta TabWrkValueSav          ; $00-$04
                        
                        ldx TabCaveShipSpeed        ; $00-$07
                        bne .DecSpeed               ; 
                        
                        lda #$00                    ; 
                        rts                         ; 
                        
.DecSpeed               dex                         ; 
                        beq CaveGetMoveValueX       ; 
                        
                        clc                         ; 
.AddNext                adc TabWrkValueSav          ; 
                        
                        dex                         ; 
                        bne .AddNext                ; 
                        
                        sta TabWrkValueSav          ; 
                        
CaveGetMoveValueX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveChkHits           Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveChkHits             subroutine                  ; 
                        lda TabCaveShipExits        ; 
                        beq .GetSprtSprtColl        ; TabCaveShipExitsNo
                        
                        rts                         ; 
                        
.GetSprtSprtColl        lda TabSprtSprtCollReg      ; 
                        ora SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        and #$01                    ; .......#
                        bne .Hit                    ; 
                        
                        rts                         ; 
                        
.Hit                    lda #$00                    ; 
                        sta TabSpriteActiveCave     ; 
                        
                        jsr ResetVoc3               ; 
                        lda #$09                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$2c                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        ldy #$80                    ; 
                        sty VCREG3                  ; SID($D412) Oscillator 3 Control
                        iny                         ; 
                        sty VCREG3                  ; SID($D412) Oscillator 3 Control
                        
                        lda #WHITE                  ; 
                        sta TabBackGroundColor      ; 
                        
.WaitBlack              lda BGCOL0                  ; VIC($D021) Background Color 0
                        and #$0f                    ; ....####
                        beq .WaitBlack              ; 
                        
                        lda #BLACK                  ; 
                        sta TabBackGroundColor      ; 
                        
                        lda #SP_Ptr_Cave_0d         ; explosion phase 1
                        sta BH_SpritePointer00_A    ; 
                        
                        lda #BH_SprtColorShipExplode; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #$02                    ; 
                        sta TabWaitTimeTOD_10th     ; 
                        jsr WaitTOD1_10th           ; 
                        
                        lda #SP_Ptr_Cave_0e         ; explosion phase 2
                        sta BH_SpritePointer00_A    ; 
                        
                        jsr WaitTOD1_10th           ; 
                        
                        lda #SP_Ptr_Cave_0f         ; explosion phase 3
                        sta BH_SpritePointer00_A    ; 
                        
                        lda #$02                    ; 
                        sta TabWaitTimeTOD_10th     ; 
                        jsr WaitTOD1_10th           ; 
                        
                        ldx #$00                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda #$09                    ; 
                        sta TabWaitTimeTOD_10th     ; 
                        jsr WaitTOD1_10th           ; 
                        
                        dec TabPlayerLives          ; 
                        bne .GetLives               ; 
                        
                        jsr ScreenStatusGameOver    ; 
                        
                        lda #TabShowScoresNo        ; 
                        sta TabPlayerShowScores     ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        sta TabP_ShowScores,x       ; 
                        
                        jsr SwitchPlayerAndValues   ; 
                        beq .DecPlayers             ; player #1
                        
                        dec TabNumberOfPlayers      ; player #2
                        pla                         ; 
                        pla                         ; 
                        jsr ScreenStatusMap         ; 
                        jmp Chapter_CaveCrossings   ; 
                        
.DecPlayers             dec TabNumberOfPlayers      ; 
                        beq .GameOver               ; 
                        
.EndCaveChapter         pla                         ; 
                        pla                         ; 
                        jmp CaveInitBay             ; 
                        
.GameOver               jmp ThreeLevel_RTS          ; Game Over
                        
.GetLives               lda TabPlayerLives          ; 
                        cmp TabPlayerShips          ; 
                        beq CaveChkHitsX            ; 
                        
                        lda TabGameMode             ; 
                        bne .IsDemo                 ; TabGameModeDemo or TabGameModeExit
                        
                        ldx #$04                    ; 
                        jsr InitTorpedoSpritePos    ; 
                        jsr CaveInitShip            ; 
                        rts                         ; 
                        
.IsDemo                 lda #TabGameModeDemoInBay   ; fleet encounters enemy fleet
                        sta TabGameMode             ; 
                        jsr ResetTOD2               ; 
                        
CaveChkHitsX            jmp CaveTryNextPlayer       ; 
; -------------------------------------------------------------------------------------------------------------- ;
CaveTryNextPlayer       subroutine                  ; 
                        jsr SpritesReset            ; common enty point
                        
                        lda #$00                    ; 
.Clear                  jsr AcToTabSpritesActive    ; 
                        
                        lda #TabCavePassedYesGame   ; 
                        sta TabPlayerCavePassed     ; 
                        
                        jsr SwitchPlayerAndValues   ; set next player
                        beq .EndCaveChapter         ; player #1 - only one player in game
                        
.SetCaveChapter         pla                         ; player #2 - restart
                        pla                         ; 
                        jmp Chapter_CaveCrossings   ; 
                        
.EndCaveChapter         pla                         ; level 1
                        pla                         ; 
CaveTryNextPlayerX      jmp CaveInitBay             ; 
; -------------------------------------------------------------------------------------------------------------- ;
CaveInitBay             subroutine                  ; 
                        lda #TabCavePassedYesGame   ; common enty point
                        sta TabPlayerCavePassed     ; 
                        jsr InitBayAreaMap          ; 
                        
                        lda #$70                    ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$00                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        lda #$78                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$00                    ; 
                        sta TabMoveSpriteLeRiTime   ; 
                        sta TabMoveSpriteUpDoTime   ; 
                        
                        jsr SpritesReset            ; 
                        
                        lda #SP_Ptr_Cave_03         ; torpedo nee
                        sta BH_SpritePointer00_A    ; 
                        
                        lda #$03                    ; ......##
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ThreeLevel_RTS        Does    : Game Over
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ThreeLevel_RTS          subroutine                  ; 
                        pla                         ; level 1
                        pla                         ; 
                        
                        pla                         ; level 2
                        pla                         ; 
                        
ThreeLevel_RTSX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveChkExitShip       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveChkExitShip         subroutine                  ; 
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$38                    ; 
                        bcc .ChkMaxPosY             ; lower
                        
.Exit                   rts                         ; 
                        
.ChkMaxPosY             lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$3e                    ; 
                        bcc .SetExitFlag            ; lower
                        
                        rts                         ; 
                        
.SetExitFlag            lda #$01                    ; 
                        sta TabCaveShipExits        ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        beq .Score                  ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .Score                  ; 
                        
                        rts                         ; 
                        
.Score                  jsr AddScore_2000           ; 
                        jsr AddScore_1000           ; 
                        
                        ldx TabPlayerShips          ; 
                        inx                         ; 
                        stx TabPlayerShips          ; 
                        cpx TabPlayerLives          ; 
                        bne .InitNextShip           ; 
                        
                        jmp CaveTryNextPlayer       ; all available ships made it through the cave
                        
.InitNextShip           jsr CaveInitShip            ; 
                        
                        ldx #$04                    ; 
                        jsr InitTorpedoSpritePos    ; 
CaveChkExitShipX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveChkExitTorpedo    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveChkExitTorpedo      subroutine                  ; 
                        ldx #$07                    ; 
.GetNextPosY            lda TabMoveSpriteUpDo,x     ; 
                        bmi .ChkMaxPosY             ; TabMoveSprite_Do
                        
.ChkMinPosY             jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        beq .NextTorpedoShot        ; 
                        
                        jmp .GetNextPosX            ; 
                        
.ChkMaxPosY             jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$b8                    ; 
                        bcs .NextTorpedoShot        ; greater/equal
                        
.GetNextPosX            lda TabMoveSpriteLeRi,x     ; 
                        bpl .ChkMaxPosX             ; TabMoveSprite_Ri
                        
                        jsr SetAcToXrTimesMult2     ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .SetNextTorpedo         ; 
                        
.ChkMinPosX             jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .NextTorpedoShot        ; 
                        
                        jmp .SetNextTorpedo         ; 
                        
.ChkMaxPosX             jsr SetAcToXrTimesMult2     ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .SetNextTorpedo         ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$50                    ; 
                        bcs .NextTorpedoShot        ; greater/equal
                        
.SetNextTorpedo         dex                         ; 
                        bne .GetNextPosY            ; 
                        
.Exit                   rts                         ; 
                        
.NextTorpedoShot        jsr InitTorpedoSpritePos    ; 
                        jsr CaveSfxTorpedoShot      ; 
CaveChkExitTorpedoX     jmp .SetNextTorpedo         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CaveDemo              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CaveDemo                subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .DemoMoveShip           ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.DemoMoveShip           lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$20                    ; 
                        bne .ChkMoveUp              ; 
                        
                        jsr ResetTOD2               ; 
                        
                        lda #TabGameModeDemoInBay   ; fleet encounters enemy fleet
                        sta TabGameMode             ; 
                        
                        pla                         ; 
                        pla                         ; 
                        rts                         ; 
                        
.ChkMoveUp              cmp #$06                    ; 
                        bne .ChkMoveRight           ; 
                        
                        lda #CIA_Joy_Up             ; 
                        sta TabJoysticAction        ; 
                        
                        rts                         ; 
                        
.ChkMoveRight           cmp #$09                    ; 
                        bne CaveDemoX               ; 
                        
                        lda #CIA_Joy_Ri             ; 
                        sta TabJoysticAction        ; 
                        
CaveDemoX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapHitMyFleetShootOut Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapHitMyFleetShootOut   subroutine                  ; 
                        lda TabBayMapSeaChapters    ; 
                        bne .TstMyFleetPosX         ; TabBayMapSeaChaptersYes
                        
.Exit                   rts                         ; 
                        
.TstMyFleetPosX         lda #$01                    ; .......#
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .TstEnemyFleetPosX      ; 
                        
.ChkHitFleet            lda SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        and #$03                    ; ......##
                        bne .StartPlaneShootOut     ; 
                        
                        rts                         ; 
                        
.TstEnemyFleetPosX      lda #$02                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .ChkHitFleet            ; 
                        
.OverlayFleetSprites    sec                         ; 
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sbc SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .ChkOverlayPosX         ; 
                        
                        eor #$ff                    ; 
                        adc #$01                    ; 
                        
.ChkOverlayPosX         cmp #BH_FleetOverlayPosX    ; 
                        bcs .Exit                   ; 
                        
                        sec                         ; 
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sbc SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        bcs .ChkOverlayPosY         ; 
                        
                        eor #$ff                    ; 
                        adc #$01                    ; 
                        
.ChkOverlayPosY         cmp #BH_FleetOverlayPosY    ; 
                        bcs .Exit                   ; 
                        
.StartPlaneShootOut     lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabMyFleetPosX          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabMyFleetMsbPosX       ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta TabMyFleetPosY          ; 
                        
                        jsr ChapterStartBell        ; 
MapHitMyFleetShootOutX  jmp Chapter_PlaneShootOut   ; Start Chapter 02 - Plane shootout
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_PlaneShootOut Does    : Chapter: Plane shootout
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_PlaneShootOut   subroutine                  ; 
                        jsr ScreenStatusSave        ; 
                        jsr ScreenStatusGetReady    ; 
                        jsr ScreenStatusRestore     ; 
                        
                        ldx #<ScreenDataOpenSea     ; 
                        ldy #>ScreenDataOpenSea     ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Sea battle (Planes)
                        jsr ClearStatusRows3_5      ; 
                        
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #$00                    ; 
.Clear                  jsr AcToTabSpritesActive    ; 
                        
                        ldx #BH_ChapterPlaneShootOut; 
                        jsr SpriteDataCopyInit      ; 
                        jsr ShootSpritesInit        ; 
                        
                        lda TabPlayerCavePassed     ; 
                        and #TabCavePassedYesGame   ; .......#
                        beq .GetSkill               ; 
                        
__ShootPlanesSkill_01   equ * + $01                 ; 
                        lda #$0a                    ; 
                        jmp .SetSkill               ; 
                        
__ShootPlanesSkill_02   equ * + $01                 ; 
.GetSkill               lda #$1e                    ; 
                        
.SetSkill               sta TabPlanes2Shoot         ; 
                        jsr CannonSpritePosInit     ; 
                        
                        lda #$b5                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$00                    ; 
                        jsr ClearGfxRow_20          ; 
                        
                        lda #$00                    ; 
                        sta TabTranspStatus         ; 
                        lda #$05                    ; 
                        sta TabPlaneFreeSprite      ; 
                        lda #$06                    ; 
                        sta TabPlaneTranspWait      ; 
                        lda #TabPlaneTranspStatusFree ; 
                        sta TabPlaneTranspStatus    ; 
                        jsr ResetVoc2               ; 
                        jsr InitVoc3                ; 
                        
                        lda #$01                    ; .......#
                        sta TabSfxToPlay            ; 
                        
.PlaneShootOut_Loop     jsr SeaPlaneFlyPhases       ; 
                        jsr SeaPlaneFlyToPlayer     ; 
                        jsr SeaPlaneStartNext       ; 
                        jsr SeaPlaneTranspChkMax    ; 
                        jsr SeaPlaneTranspChkHit    ; 
                        jsr SeaPlaneTranspChkExpl   ; 
                        jsr CannonPlayerMove        ; 
                        jsr CannonPlayerShootInit   ; 
                        jsr GetSprtSprtCollReg      ; sprite/sprite collision
                        jsr CannonPlayerShoot       ; 
                        jsr PlayerAmmoInc           ; 
                        jsr SeaPlaneChkHit          ; 
                        jsr SeaPlaneExplodeOff      ; 
                        
.ChkFinish              lda TabPlanes2Shoot         ; 
                        ora TabTargetHit            ; 
                        bne .PlaneShootOut_Next     ; 
                        
                        jsr SwitchPlayerAndValues   ; 
                        beq .GoShipShootOut         ; player #1
                        
.GoPlaneShootOut        jmp Chapter_PlaneShootOut   ; player #2
.GoShipShootOut         jmp Chapter_ShipShootOut    ; 
                        
.PlaneShootOut_Next     jsr SeaPlaneShootInit       ; 
                        jsr SeaPlaneShoot           ; 
                        jsr SeaPlaneDamageOut       ; check for game over - leave then
                        jsr SeaPlaneSoundTurn       ; 
                        jsr SeaPlaneSoundShoot      ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr ScreenStatusAmmoFlack   ; 
                        jsr ScreenStatusShowShips   ; 
                        jsr CheckGamePause          ; on C= commodore key
                        jsr CheckGameFinished       ; idle time/death - 3 level return to BASIC via GameStart
                        jsr ScreenStatusDestroy     ; 
                        jsr CheckGameIdleTime       ; 
                        jsr SeaPlaneDemo            ; 
                        
Chapter_PlaneShootOutX  jmp .PlaneShootOut_Loop     ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShootSpritesInit      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShootSpritesInit        subroutine                  ; 
                        jsr GameInit                ; 
                        
                        lda #BH_SprtColorShipCannon ; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #BH_SprtColorShipBullet ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        
                        lda #$0b                    ; 
                        sta TabVoc2AttDec           ; 
                        lda #$1c                    ; 
                        sta TabVoc2FreqHi           ; 
                        
                        lda #$00                    ; 
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta TabPlayerShoots         ; TabPlayerShootsNo
                        sta TabBulletDamage         ; 
                        sta TabTargetHit            ; TabTargetHitNo
                        sta BH_SpritePointer04_A    ; 
                        sta TabBulletActive         ; TabBulletActiveNo
                        sta TabSeaShipMsgDestroy    ; TabSeaShipMsgDestroyNo
                        jsr SpritesReset            ; 
                        
                        lda #$0a                    ; 
                        sta TabPlayerAmmo           ; 
                        
                        lda #$01                    ; 
                        sta TabPlayerShootSpriteNo  ; 
                        
                        lda #SP_Ptr_Ship_24         ; water splash phase 1 - min
                        sta TabPlrShootSprtBasePtr1 ; 
                        
                        lda #SP_Ptr_Ship_30         ; enemy water splash phase 10
                        sta TabPlrShootSprtBasePtr2 ; 
                        
                        lda #TabSpriteStatusExpl    ; 
                        sta TabTranspStatus         ; 
                        sta TabBulletStatus1        ; 
                        sta TabBulletStatus2        ; 
                        
ShootSpritesInitX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonPlayerMove      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonPlayerMove        subroutine                  ; 
.ChkNextMoveLeRi        lda TabWaitFlagMoveLeRi     ; 
                        bne .ChkNextMoveUpDo        ; TabWaitFlag_Yes
                        
.SetWaitLeRi            inc TabWaitFlagMoveLeRi     ; TabWaitFlag_Yes
                        
.Chk_Le                 lda #CIA_Joy_Le             ; .....#..
                        bit TabJoysticAction        ; 
                        beq .Chk_Ri                 ; 
                        
.MovCannonLe            jsr ClearDirInJoyAction     ; 
                        
.ChkMsbPosX             lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$01                    ; .......#
                        bne .DecPosX                ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
.MinLe                  cmp #$18                    ; 
                        beq .ChkNextMoveUpDo        ; 
                        
.DecPosX                lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$01                    ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .ChkNextMoveUpDo        ; 
                        
.SetMsbPosX             lda #$01                    ; 
                        eor MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .ChkNextMoveUpDo        ; 
                        
.Chk_Ri                 lda #CIA_Joy_Ri             ; ....#...
                        bit TabJoysticAction        ; 
                        beq .ChkNextMoveUpDo        ; 
                        
.MovCannonRi            jsr ClearDirInJoyAction     ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$01                    ; 
                        beq .IncPosX                ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
.MaxRi                  cmp #$40                    ; 
                        beq .ChkNextMoveUpDo        ; 
                        
.IncPosX                inc SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bne .ChkNextMoveUpDo        ; 
                        
                        jmp .SetMsbPosX             ; 
                        
.ChkNextMoveUpDo        lda TabWaitFlagMoveUpDo     ; 
                        beq .SetWaitUpDo            ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.SetWaitUpDo            inc TabWaitFlagMoveUpDo     ; TabWaitFlag_Yes
                        
.Chk_Do                 lda #CIA_Joy_Do             ; ......#.
                        bit TabJoysticAction        ; 
                        beq .Chk_Up                 ; 
                        
.MovCannonDo            jsr ClearDirInJoyAction     ; 
                        
                        lda TabCannonShotDegree     ; 
.MinDo                  cmp #$20                    ; 
                        beq .GetSpritePtr           ; 
                        
                        dec TabCannonShotDegree     ; 
                        jmp .GetSpritePtr           ; 
                        
.Chk_Up                 lda #CIA_Joy_Up             ; .......#
                        bit TabJoysticAction        ; 
                        beq .GetSpritePtr           ; 
                        
.MovCannonUp            jsr ClearDirInJoyAction     ; 
                        
                        lda TabCannonShotDegree     ; 
.MaxUp                  cmp #$99                    ; 
                        beq .GetSpritePtr           ; 
                        
                        inc TabCannonShotDegree     ; 
                        
.GetSpritePtr           lda TabCannonShotDegree     ; 
                        sec                         ; 
                        sbc #$20                    ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        clc                         ; 
                        adc TabPlrShootSprtBasePtr2 ;  
                        sta BH_SpritePointer00_A    ; 
                        
CannonPlayerMoveX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonPlayerShootInit Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonPlayerShootInit   subroutine                  ; 
                        lda #CIA_Joy_Fi             ; ...#....
                        bit TabJoysticAction        ; 
                        bne .ClrFireButton          ; 
                        
.Exit                   rts                         ; 
                        
.ClrFireButton          jsr ClearDirInJoyAction     ; 
                        
                        lda TabPlayerShoots         ; 
                        bne .Exit                   ; TabPlayerShootsYes
                        
                        lda TabPlayerAmmo           ; 
                        beq .Exit                   ; 
                        
                        ldx TabPlayerShootSpriteNo  ; 
                        lda TabSpriteStatus,x       ; 
                        bne .Exit                   ; TabSpriteStatusUsed
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$ba                    ; 
                        sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        sec                         ; 
                        sbc TabCannonShotDegree     ; 
                        sta TabBulletHight,x        ; 
                        
                        lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ClrMsbPosX             ; 
                        
.SetMsbPosX             jsr SetAcToXrTimesMult2     ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .GetSpritePtrBase       ; 
                        
.ClrMsbPosX             jsr SetAcToXrTimesMult2     ; 
                        eor #$ff                    ; 
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.GetSpritePtrBase       lda TabPlrShootSprtBasePtr1 ; 
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #$00                    ; 
                        sta GR_SpriteStepCount,x    ; 
                        
                        cpx #$01                    ; 
                        bne .GetSpriteNo_1          ; 
                        
.GetSpriteNo_2          lda #$02                    ; 
                        jmp .SetSpriteNo            ; 
                        
.GetSpriteNo_1          lda #$01                    ; 
                        
.SetSpriteNo            sta TabPlayerShootSpriteNo  ; 
                        
                        lda #TabSpriteStatusUsed    ; 
                        sta TabSpriteStatus,x       ; 
                        jsr EnableSpriteNoFromXr    ; 
                        jsr SetVoc2_Cannon          ; 
                        jsr PlayerAmmoDec           ; 
                        jsr CannonPlayerSetSprtPtr  ; 
                        
                        lda #TabPlayerShootsYes     ; 
                        sta TabPlayerShoots         ; 
                        
                        lda #$00                    ; 
                        sta TabPlayerBulletPhaseNo  ; 
                        
CannonPlayerFireX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonPlayerSetSprtPtr Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonPlayerSetSprtPtr  subroutine                  ; 
                        lda BH_SpritePointer00_A    ; 
                        clc                         ; 
                        adc #$02                    ; 
                        
                        pha                         ; save
                        sbc #$0f                    ; 
                        cmp TabPlrShootSprtBasePtr2 ; 
                        pla                         ; restore
                        bcs CannonPlayerSetSprtPtrX ; greater/equal
                        
                        sta BH_SpritePointer00_A    ; lower
                        
.SetInitialWait         lda #$0a                    ; 
                        sta TabWaitTimeMoveUpDo     ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveUpDo     ; 
                        
CannonPlayerSetSprtPtrX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonPlayerShoot     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonPlayerShoot       subroutine                  ; 
                        ldx #$02                    ; 
.GetStatusBullets       lda TabSpriteStatus,x       ; 
                        beq .SetNextBullet          ; TabSpriteStatusFree
                        
                        lda TabWaitFlagMoveLeRi,x   ; 
                        bne .SetNextBullet          ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagMoveLeRi,x   ; TabWaitFlag_Yes
                        inc GR_SpriteStepCount,x    ; 
                        
                        lda GR_SpriteStepCount,x    ; 
                        cmp #$10                    ; 
                        bne .ChkBullet              ; 
                        
                        lda #TabPlayerShootsNo      ; 
                        sta TabPlayerShoots         ; 
                        
.ChkBullet              lda GR_SpriteStepCount,x    ; 
                        cmp #$1e                    ; max
                        beq .ClrBullet              ; 
                        
.SetSpritePtr1          lda TabPlrShootSprtBasePtr1 ; 
                        clc                         ; 
__PlayerShootSprtPtrAdd equ * + $01                 ; 
                        adc #$09                    ; 
                        cmp BH_SpritePointers_A,x   ; 
                        beq .IncBulletFlightHeight  ; 
                        
.SetSpritePtr2          lda GR_SpriteStepCount,x    ; 
                        lsr a                       ; 
                        clc                         ; 
                        adc TabPlrShootSprtBasePtr1 ; 
                        sta BH_SpritePointers_A,x   ; 
                        
.IncBulletFlightHeight  lda TabBulletHight,x        ; 
                        lsr a                       ; /2
                        lsr a                       ; /4
                        lsr a                       ; /8
                        sta TabWrkPlayerShot        ; 
                        
                        lda TabBulletHight,x        ; 
                        sec                         ; 
                        sbc TabWrkPlayerShot        ; 
                        sta TabBulletHight,x        ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc TabWrkPlayerShot        ; 
                        sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
.SetNextBullet          dex                         ; 
                        bne .GetStatusBullets       ; 
                        
                        rts                         ; 
                        
.ClrBullet              lda #TabSpriteStatusFree    ; 
                        sta TabSpriteStatus,x       ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda #$01                    ; 
                        sta TabPlayerBulletPhaseNo  ; 
CannonPlayerShootX      jmp .SetNextBullet          ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneChkHit        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneChkHit          subroutine                  ; 
                        ldx #$05                    ; 
.ChkHitTarget           lda TabTargetHit            ; 
                        beq .GetPlaneDistance       ; TabTargetHitNo
                        
                        rts                         ; 
                        
.GetPlaneDistance       lda TabPlaneDistance,x      ; 
.ChkPlaneDistanceMin    cmp #$04                    ; 
                        bcs .ChkPlaneDistanceMax    ; greater/equal
                        
.SetNextPlaneHi         inx                         ; lower
                        cpx #$08                    ; 
                        bne .ChkHitTarget           ; 
                        
                        rts                         ; 
                        
.ChkPlaneDistanceMax    cmp #$06                    ; 
                        bcs .SetNextPlaneHi         ; greater/equal
                        
                        jsr SetAcToXrTimesMult2     ; lower
                        bit TabSprtSprtCollReg      ; 
                        bne .PlaneHit               ; 
                        
                        lda #$10                    ; ...#....
                        bit TabSprtSprtCollReg      ; 
                        bne .PlaneHit               ; 
                        
.Exit                   rts                         ; 
                        
.PlaneHit               stx TabSavPlaneSpriteNo     ; save plane sprite number
                        
                        ldx #$01                    ; 
.GetNextPlaneLo         jsr SetAcToXrTimesMult2     ; 
                        bit TabSprtSprtCollReg      ; 
                        bne .ChkStep                ; 
                        
.SetNextPlaneLo         inx                         ; 
                        cpx #$03                    ; 
                        bne .GetNextPlaneLo         ; 
                        
                        rts                         ; 
                        
.ChkStep                lda GR_SpriteStepCount,x    ; 
                        cmp #$12                    ; 
                        bcc .SetNextPlaneLo         ; lower
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        bcs .ChkCenter              ; 
                        
                        eor #$ff                    ; make positve
                        clc                         ; 
                        adc #$01                    ; 
                        
.ChkCenter              cmp #$08                    ; 
                        bcs .SetNextPlaneLo         ; 
                        
                        lda #$00                    ; 
                        sta TabSpriteStatus,x       ; TabSpriteStatusFree
                        sta TabPlayerShoots         ; TabPlayerShootsNo
                        jsr DisableSpriteNoFromXr   ; 
                        jsr PlaneExplode            ; 
                        
.Score                  jsr AddScore_400            ; 
                        
                        dec TabPlanes2Shoot         ; 
SeaPlaneChkHitX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; PlaneExplode          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
PlaneExplode            subroutine                  ; 
                        ldx TabSavPlaneSpriteNo     ; 
                        
                        lda #$00                    ; 
                        sta TabSpriteActivePlane,x  ; TabSpriteActiveNo
                        sta TabPlaneDirReLi,x       ; TabPlaneDirReLiStop
                        sta TabPlaneDirUpDo,x       ; 
                        sta TabPlaneVal_1           ; 
                        sta TabPlaneVal_2           ; 
                        sta TabPlaneVal_3           ; 
                        
                        lda #TabPlaneDist_Explode   ; 
                        sta TabPlaneDistance,x      ; 
                        sta TabPlaneDistance + $04  ; 
                        
                        jsr SetAcToXrTimesMult2     ; 
                        ora #$10                    ; ...#....
                        ora SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BH_SprtColorPlaneExpl_MC0 ; 
                        sta SPMC0                   ; VIC($D025) Sprite Multicolor Register 0
                        
                        lda #BH_SprtColorPlaneExpl_MC1 ; 
                        sta SPMC1                   ; VIC($D026) Sprite Multicolor Register 1
                        
                        lda #BH_SprtColorPlaneExplode ; 
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
                        lda #SP_Ptr_Plane_49        ; explode plane left
                        sta BH_SpritePointers_A,x   ; 
                        
                        lda #SP_Ptr_Plane_4a        ; explode plane right
                        sta BH_SpritePointer04_A    ; 
                        
                        lda #$10                    ; 
                        sta TabPlaySfxVoc1_03       ; 
                        
                        lda #$01                    ; 
                        sta TabWaitFlagExplode      ; TabWaitFlag_Yes
                        sta TabTargetHit            ; TabTargetHitYes
                        
                        jsr InitVoc1                ; 
                        
                        lda #$f7                    ; ####.###
                        and TabSfxToPlay            ; 
                        ora #$01                    ; .......#
                        sta TabSfxToPlay            ; 
                        
                        jsr InitVoc3                ; 
PlaneExplodeX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneExplodeOff    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneExplodeOff      subroutine                  ; 
                        lda TabTargetHit            ; 
                        bne .ChkExplode             ; TabTargetHitYes
                        
.Exit                   rts                         ; 
                        
.ChkExplode             lda TabWaitFlagExplode      ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        ldx #$04                    ; TabWaitFlag_No
                        jsr DisableSpriteNoFromXr   ; 
                        
                        ldx TabSavPlaneSpriteNo     ; 
                        jsr DisableSpriteNoFromXr   ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        ora #$10                    ; ...#....
                        eor #$ff                    ; ########
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$00                    ; 
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        sta TabTargetHit            ; TabTargetHitNo
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
                        lda #$ff                    ; 
                        sta TabSpriteStatus,x       ; TabSpriteStatusExpl
                        sta TabFlagTankEneMove      ; 
                        
SeaPlaneExplodeOffX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneShootInit     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneShootInit       subroutine                  ; 
                        lda TabBulletActive         ; 
                        beq .ChkSprtPtr04           ; TabBulletActiveNo
                        
.Exit                   rts                         ; 
                        
.ChkSprtPtr04           lda BH_SpritePointer04_A    ; 
                        cmp #SP_Ptr_Plane_10        ; plane double sprite phase 5 right - last before turn away
                        bne .Exit                   ; 
                        
                        cmp #SP_Ptr_Plane_11        ; plane turn away mid phase 1 left
                        bcs .Exit                   ; greater/equal
                        
                        lda SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$0c                    ; 
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .ChkBit4                ; 
                        
.ClrPosXMSB_03          lda #$f7                    ; ####.###
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .IncSprt4PosY           ; 
                        
.ChkBit4                lda #$10                    ; ...#....
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ClrPosXMSB_03          ; 
                        
.SetPosXMSB_03          lda #$08                    ; ....#...
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.IncSprt4PosY           lda SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        clc                         ; 
                        adc #$0b                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #TabSpriteStatusUsed    ; 
                        sta TabBulletStatus         ; 
                        
                        ldx #$03                    ; ......##
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$01                    ; 
                        sta TabWaitFlagBulletFly    ; TabWaitFlag_Yes
                        sta TabWaitBulletFlySav     ; 
                        sta TabWaitBulletFly        ; 
                        sta TabBulletActive         ; TabBulletActiveYes
                        
                        lda #SP_Ptr_Plane_2b        ; bullet phase  7
                        sta BH_SpritePointer03_A    ; 
                        
                        lda #$00                    ; 
                        sta GR_BulletStepCount      ; 
                        
                        lda #BH_SprtColorPlaneBullet; 
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        
SeaPlaneShootInitX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneShoot         Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneShoot           subroutine                  ; 
                        lda TabBulletActive         ; 
                        cmp #TabBulletActiveYes     ; 
                        beq .FlyBullet              ; 
                        
.Exit                   rts                         ; 
                        
.FlyBullet              lda TabWaitFlagBulletFly    ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagBulletFly    ; TabWaitFlag_Yes
                        
.ChkBulletPosX          lda SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        cmp #$b4                    ; 
                        bcs .Hit                    ; greater/equal
                        
                        clc                         ; 
                        adc #$04                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        inc GR_BulletStepCount      ; 
                        lda GR_BulletStepCount      ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        eor #$ff                    ; 
                        sec                         ; 
                        adc #$6b                    ; 
                        cmp #SP_Ptr_Plane_24        ; bullet phase  1 - max
                        bcc .Exit_2                 ; lower
                        
                        sta BH_SpritePointer03_A    ; 
                        
.Exit_2                 rts                         ; 
                        
.Hit                    dec TabBulletActive         ; TabBulletActiveNo
                        
                        lda #TabSpriteStatusFree    ; 
                        sta TabBulletStatus         ; 
                        
.DisapleBullet          ldx #$03                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        inc TabBulletDamage         ; 
SeaPlaneShootX          rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneDamageOut     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneDamageOut       subroutine                  ; 
                        lda TabSeaShipMsgDestroy    ; 
                        beq .GetDamageCount         ; TabSeaShipMsgDestroyNo
                        
                        rts                         ; 
                        
.GetDamageCount         lda TabBulletDamage         ; 
                        jsr MakeDecimal             ; xr=100 yr=10_1
                        
                        tya                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffDamage1 ; 
                        stx BH_Scrn_Stat + BH_Scrn_Stat_OffDamage10 ; 
                        
                        lda #WHITE                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffDamage1 ;
                        sta COLORAM + BH_Scrn_Stat_OffDamage10 ;
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDamage ; 
                        ldx #<TabTextStatusDamage   ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
                        lda TabBulletDamage         ; 
__ShootPlanesDamage     equ * + $01                 ; 
                        cmp #$14                    ; 
                        beq .Destroyed              ; maximum reached
                        
                        rts                         ; 
                        
.Destroyed              jsr ResetTOD1               ; 
                        
                        lda #TabSeaShipMsgDestroyYes; 
                        sta TabSeaShipMsgDestroy    ; 
                        
                        dec TabPlayerLives          ; 
.ChkGameOver            lda TabPlayerLives          ; 
                        bne .InitDamageCount        ; 
                        
                        jsr ScreenStatusGameOver    ; 
                        
                        lda #TabShowScoresNo        ; 
                        sta TabPlayerShowScores     ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        sta TabP_ShowScores,x       ; 
                        jsr SwitchPlayerAndValues   ; 
                        beq .ContWithPlayer_01      ; player #1
                        
.ContWithPlayer_02      dec TabNumberOfPlayers      ; player #2
                        
                        pla                         ; 
                        pla                         ; 
                        jsr ScreenStatusMap         ; 
                        jsr ClearStatusRows3_5      ; 
                        jmp Chapter_PlaneShootOut   ; 
                        
.ContWithPlayer_01      dec TabNumberOfPlayers      ; 
                        beq .AllPlayerGone          ; 
                        
                        pla                         ; 
                        pla                         ; 
                        jmp Chapter_ShipShootOut    ; 
                        
.AllPlayerGone          jmp ThreeLevel_RTS          ; restart game
                        
.InitDamageCount        lda #$00                    ; 
                        sta TabBulletDamage         ; 
                        
SeaPlaneDamageOutX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaPlaneDemo          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaPlaneDemo            subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .ChkDemoTime            ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.ChkDemoTime            lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$40                    ; 
                        bne .IniPlaneDistance       ; 
                        
.DemoFinished           jsr ResetTOD2               ; 
                        
                        pla                         ; 
                        pla                         ; 
                        jmp Chapter_ShipShootOut    ; 
                        
.IniPlaneDistance       ldx #$05                    ; 
.ChkPlaneDistance       lda TabPlaneDistance,x      ; 
                        cmp #$04                    ; near enough
                        bcs .GetSpriteOff           ; yes - greater/equal
                        
                        inx                         ; 
                        cpx #$08                    ; 
                        bne .ChkPlaneDistance       ; 
                        
.Exit                   rts                         ; no planes in the near
                        
.GetSpriteOff           jsr SetYrToXrMult2          ; yr = xr * 2
                        lda TabCannonShotDegree     ; 
                        cmp SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        beq .ChkPosX_MSB            ; equal
                        bcc .GetJoyUp               ; lower
                        
.GetJoyDo               lda #CIA_Joy_Do             ; greater
                        sta TabJoysticAction        ; 
                        
                        jmp .ChkPosX_MSB            ; 
                        
.GetJoyUp               lda #CIA_Joy_Up             ; 
                        sta TabJoysticAction        ; 
                        
.ChkPosX_MSB            lda #$10                    ; ...#....
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .SetPosX_MSB            ; 
                        
                        lda #$01                    ; .......#
                        
.SetPosX_MSB            sta TabDemoPlanePosX_MSB    ; 
                        
                        lda #$01                    ; .......#
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta TabDemoPlaneMyPosX_MSB  ; 
                        
                        lda SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabDemoPlanePosX        ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta TabDemoPlaneMyPosX      ; 
                        
                        ldy #$03                    ; 
.ChkPlanePosX           dey                         ; 
                        beq .GetJoyFi               ; 
                        
                        lda [TabDemoPlanePosX   - $01],y ; 
                        cmp [TabDemoPlaneMyPosX - $01],y ; 
                        beq .ChkPlanePosX           ; equal
                        bcs .GetJoyRi               ; greater
                        
.GetJoyLe               lda #CIA_Joy_Le             ; lower
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
                        jmp .GetJoyFi               ; 
                        
.GetJoyRi               lda #CIA_Joy_Ri             ; 
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
.GetJoyFi               lda #CIA_Joy_Fi             ; 
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
SeaPlaneDemoX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearStatusRows3_5    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearStatusRows3_5      subroutine                  ; 
                        lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapAerial ; 
                        ldx #BH_Scrn_Stat_LenMapAerial ; 
                        jsr ScreenStatusTextOutRow5 ; xr=lenght blank out yr=output text offset
                        
                        ldy #BH_Scrn_Stat_OffAmmoShipTxt ; 
                        ldx #BH_Scrn_Stat_LenAmmoShipTxt + BH_Scrn_Stat_LenAmmoShipCube ; 
                        jsr ScreenStatusTextOutRow3 ; xr=lenght blank out  yr=output text offset
ClearStatusRows3_5X     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_ShipShootOut  Does    : Chapter: Ship shootout
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_ShipShootOut    subroutine                  ; 
                        jsr ScreenStatusSave        ; 
                        jsr ScreenStatusGetReady    ; 
                        jsr ScreenStatusRestore     ; 
                        
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        ldx #<ScreenDataOpenSea     ; 
                        ldy #>ScreenDataOpenSea     ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Sea battle (Ships)
                        jsr ScreenStatusInitShip    ; 
                        
                        ldx #BH_ChapterShipShootOut ; 
                        jsr SpriteDataCopyInit      ; 
                        jsr SeaShipShootInit        ; 
                        jsr CannonSpritePosInit     ; 
                        
                        lda #$b5                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        jsr ClearGfxRow_20          ; 
                        
                        lda #$03                    ; 
                        sta TabEnemyBulletPhase     ; 
                        
                        jsr ShipInitCarrier         ; 
                        jsr GetRND                  ; 
.LimitDegree            cmp #$0d                    ; 
                        bcc .SetStartDegree         ; lower
                        
                        sbc #$0d                    ; greater/equal
                        jmp .LimitDegree            ; 
                        
.SetStartDegree         sta TabShipCannonDegree     ; 
                        
.ShipShootOut_Loop      jsr SeaShipEnemyBulletInit  ; 
                        jsr SeaShipEnemyGetCollSpBg ; 
                        jsr SeaShipEnemyBullet      ; 
                        jsr CannonPlayerMove        ; 
                        jsr CannonPlayerShootInit   ; 
                        
                        lda #$01                    ; 
                        sta TabPlayerShootSpriteNo  ; 
                        
                        jsr CannonPlayerBullet      ; 
                        jsr PlayerAmmoInc           ; 
                        
.ChkEndOfChapter        lda TabTargetShipHit        ; 
                        eor #$ff                    ; ########
                        ora TabPlayerVesselActive   ; one bit per ship
                        ora TabBulletActive         ; 
                        ora TabSeaShipMsgDestroy    ; 
                        ora TabP_ShotStatus         ; 
                        bne .ShipShootOut_Next      ; 
                        
.ReadyNextChapter       jsr SpritesReset            ; 
                        jsr SeaShipReduceCount      ; 
                        jsr SwitchPlayerAndValues   ; 
                        beq .GoInitBayArea          ; player #1
                        
.GoShipShootOut         jmp Chapter_ShipShootOut    ; player #2
.GoInitBayArea          jmp InitBayArea             ; with RTS
                        
.ShipShootOut_Next      jsr SeaShipMoveCarrier      ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr ScreenStatusShowDegree  ; 
                        jsr ScreenStatusAmmoShip    ; 
                        jsr SeaShipEneBulletSfxIni  ; 
                        jsr SeaShipEneBulletSfxMod  ; 
                        jsr SeaShipStatusShipsOut   ; 
                        jsr CheckGamePause          ; on C= commodore key
                        jsr SeaShipChkDestroyMsg    ; 
                        jsr BulletWaterSplashOff    ; 
                        jsr CheckGameFinished       ; idle time/death - 3 level return to BASIC via GameStart
                        jsr SeaShipSinkAndScore     ; 
                        jsr SeaShipEneExplodeOff    ; 
                        jsr CheckGameIdleTime       ; 
                        jsr SeaShipDemo             ; 
                        
Chapter_ShipShootOutX   jmp .ShipShootOut_Loop      ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipShootInit      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipShootInit        subroutine                  ; 
                        jsr GameInit                ; 
                        
                        lda #$9c                    ; 
                        sta TabBulletEnemyDist      ; 
                        
                        lda #SP_Ptr_Ship_01         ; cannon shot phase  1 - max
                        sta TabPlrShootSprtBasePtr1 ; 
                        
                        lda #TabPlayerShootsNo      ; 
                        sta TabPlayerShoots         ; 
                        
                        lda #SP_Ptr_Ship_0d         ; cannon down  1 - max up
                        sta TabPlrShootSprtBasePtr2 ; 
                        
                        lda #$00                    ; 
                        sta TabBulletActive         ; TabBulletActiveNo
                        sta TabJoysticAction        ; 
.Clear                  jsr AcToTabSpritesActive    ; 
                        
                        lda #TabTargetHitNo         ; 
                        sta TabTargetHit            ; 
                        
                        lda #TabTargetShipHitNo     ; 
                        sta TabTargetShipHit        ; hit water
                        
                        lda #$10                    ; 
                        sta TabWaitBulletFlySav     ; 
                        sta TabWaitBulletFly        ; 
                        jsr SpritesReset            ; 
                        
                        lda #$20                    ; ..#.....
                        bit TabPlayerVesselActive   ; one bit per ship
                        beq .GetZero                ; 
                        
.GetOne                 lda #TabFlagCarrierActiveYes; 
                        jmp .SetValue               ; 
                        
.GetZero                lda #TabFlagCarrierActiveNo ; 
                        
.SetValue               sta TabFlagCarrierActive    ; 
                        
                        lda #$0a                    ; 
                        sta TabPlayerAmmo           ; 
                        
                        lda #$0c                    ; 
                        sta TabVoc2AttDec           ; 
                        lda #$08                    ; 
                        sta TabVoc2FreqHi           ; 
                        lda #$f3                    ; 
                        sta RESON                   ; SID($D417) Filter Resonance Ctrl
                        
SeaShipShootInitX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonSpritePosInit   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonSpritePosInit     subroutine                  ; 
                        lda TabPlrShootSprtBasePtr2 ; 
                        clc                         ; 
                        adc #$08                    ; 
                        sta BH_SpritePointer00_A    ; 
                        
                        lda #$60                    ; 
                        sta TabCannonShotDegree     ; 
                        
                        lda #$a0                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$ac                    ; 
                        sta SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$fe                    ; #######.
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$01                    ; 
                        sta TabWaitTimeCanLeRiSav   ; 
                        sta TabWaitTimeCanUpDoSav   ; 
                        
                        ldx #$00                    ; 
                        jsr EnableSpriteNoFromXr    ; 
CannonSpritePosInitX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEnemyBulletInit Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEnemyBulletInit  subroutine                  ; 
                        lda TabBulletActive         ; 
                        beq .ChkWait                ; TabBulletActiveNo
                        
.Exit                   rts                         ; 
                        
.ChkWait                lda TabWaitFlagBulletFly    ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        lda #$fb                    ; #####.##
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda TabEnemyBulletPhase     ; 
.ChkPase_01             cmp #$01                    ; 
                        bne .ChkPase_02             ; 
                        
                        lda #$33                    ; 
                        bne .SetNewPosX             ; always
                        
.ChkPase_02             cmp #$02                    ; 
                        bne .ChkPase_03             ; always
                        
                        lda #$6e                    ; 
                        bne .SetNewPosX             ; always
                        
.ChkPase_03             cmp #$03                    ; 
                        bne .ChkPase_04             ; always
                        
                        lda #$ff                    ; 
                        bne .SetNewPosX             ; always
                        
.ChkPase_04             cmp #$04                    ; 
                        bne .Exit                   ; 
                        
                        lda #$04                    ; .....#..
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$32                    ; 
                        
.SetNewPosX             sta SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$01                    ; 
                        sta TabPlaneDistance        ; 
                        sta TabBulletActive         ; TabBulletActiveYes
                        sta TabBulletStatus         ; TabBulletStatusUsed
                        
                        lda #$8c                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
                        lda #$10                    ; ...#....
                        bit TabSfxToPlay            ; 
                        bne .SetSpritePtr           ; 
                        
                        jsr ResetVoc1               ; 
                        
                        lda #$9e                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$0c                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
.SetSpritePtr           lda #SP_Ptr_Ship_1d         ; enemy bullet phase 1 - min
                        sta BH_SpritePointer02_A    ; 
                        
                        lda #BLACK                  ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        
                        ldx #$02                    ; 
                        jsr EnableSpriteNoFromXr    ; 
SeaShipEnemyBulletInitX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEnemyBullet    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEnemyBullet      subroutine                  ; 
                        lda TabBulletActive         ; 
                        cmp #TabBulletActiveYes     ; 
                        beq .ChkDistances           ; 
                        
                        rts                         ; 
                        
.ChkDistances           lda TabPlaneDistance        ; 
                        cmp #$01                    ; 
                        beq .Dist_01                ; 
                        
                        cmp #$02                    ; 
                        beq .Dist_02                ; 
                        
                        cmp #$03                    ; 
                        beq .Dist_03                ; 
                        
                        cmp #$04                    ; 
                        bne .Exit                   ; 
                        
                        jmp .Dist_04                ; 
                        
.Exit                   rts                         ; 
                        
.Dist_01                lda SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        cmp #$18                    ; 
                        bcc B_7a40                  ; lower
                        
                        lda TabWaitFlagPlaneFly     ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagPlaneFly     ; TabWaitFlag_Yes
                        
                        lda SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc #$02                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
.Dist_01X               rts                         ; 
                        
.Dist_03                lda BH_SpritePointer02_A    ; 
                        cmp #SP_Ptr_Ship_2b         ; enemy water splash phase  5
                        beq B_7ab9                  ; 
                        
                        lda TabWaitFlagPlaneFly     ; 
                        bne .Dist_03X               ; TabWaitFlag_Yes
                        
                        lda TabWaitFlagBullFlyESav  ; 
                        asl a                       ; 
                        sta TabWaitFlagBullFlyESav  ; 
                        sta TabWaitFlagBullFlyE     ; 
                        
                        inc TabWaitFlagPlaneFly     ; TabWaitFlag_Yes
                        inc BH_SpritePointer02_A    ; 
                        
.Dist_03X               rts                         ; 
                        
B_7ab9                  lda #$04                    ; 
                        sta TabPlaneDistance        ; 
                        
                        lda #$0e                    ; 
                        sta TabWaitFlagBullFlyESav  ; 
                        sta TabWaitFlagBullFlyE     ; 
                        
                        rts                         ; 
                        
B_7a40                  lda #$02                    ; 
                        sta TabPlaneDistance        ; 
                        
                        lda #$d0                    ; 
                        sta TabWaitFlagBullFlyE     ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagPlaneFly     ; 
                        
                        lda #SP_Ptr_Ship_1f         ; enemy bullet phase 3
                        sta BH_SpritePointer02_A    ; 
                        
                        rts                         ; 
                        
.Dist_02                lda SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        cmp TabBulletEnemyDist      ; 
                        bcs .ChkBulletHit           ; greater/equal
                        
                        lda TabWaitFlagPlaneFly     ; lower
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagPlaneFly     ; TabWaitFlag_Yes
                        
                        lda SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        clc                         ; 
                        adc #$08                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
.Dist_02X               rts                         ; 
                        
.ChkBulletHit           lda TabBulletEnemyDist      ; 
                        cmp #$b0                    ; 
                        bcc .BulletHitsWater        ; lower
                        
                        jmp .BulletHitsShip         ; 
                        
.BulletHitsWater        lda #$03                    ; 
                        sta TabPlaneDistance        ; 
                        
                        lda #$02                    ; 
                        sta TabWaitFlagBullFlyESav  ; 
                        
                        lda #SP_Ptr_Ship_27         ; enemy water splash phase  1 - start
                        sta BH_SpritePointer02_A    ; 
                        
                        lda #$04                    ; 
                        ora YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        
                        lda #WHITE                  ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        
                        lda SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc #$15                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
                        lda #$cf                    ; ##..####
                        and TabSfxToPlay            ; 
                        sta TabSfxToPlay            ; 
                        
                        jsr ResetVoc3               ; 
                        
                        lda #$24                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        lda #$4c                    ; 
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        lda #$81                    ; 
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
.ExitBulletHitsWater    rts                         ; 
                        
.Dist_04                lda TabWaitFlagPlaneFly     ; 
                        bne .Dist_04X               ; TabWaitFlag_Yes
                        
                        lda BH_SpritePointer02_A    ; 
                        cmp #SP_Ptr_Ship_31         ; enemy water splash phase 11 - end
                        beq .ClrBullet              ; 
                        
                        lda TabWaitFlagBullFlyESav  ; 
                        sec                         ; 
                        sbc #$02                    ; 
                        sta TabWaitFlagBullFlyESav  ; 
                        sta TabWaitFlagBullFlyE     ; 
                        
                        inc TabWaitFlagPlaneFly     ; TabWaitFlag_Yes
                        inc BH_SpritePointer02_A    ; 
                        
.Dist_04X               rts                         ; 
                        
.ClrBullet              lda #$00                    ; BLACK/TabPlaneDist_Ini
                        sta TabBulletActive         ; TabBulletActiveNo
                        sta TabPlaneDistance        ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        sta TabBulletStatus         ; TabBulletStatusInit
                        
                        lda #$01                    ; 
                        sta TabWaitFlagBulletFly    ; TabWaitFlag_Yes
                        sta TabWaitFlagBullFlyESav  ; 
                        sta TabWaitFlagBullFlyE     ; 
                        
                        lda #$fb                    ; #####.##
                        and YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        
                        ldx #$02                    ; ......#.
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda TabBulletEnemyDist      ; 
                        clc                         ; 
                        
__DistSeaShipSkill      equ * + $01                 ; 
                        adc #$04                    ; 
                        sta TabBulletEnemyDist      ; 
                        
.ChkCavePassed          lda #TabCavePassedYesGame   ; 
                        bit TabPlayerCavePassed     ; 
                        bne .ChkMax                 ; 
                        
                        lda #$06                    ; 
                        clc                         ; 
                        adc TabBulletEnemyDist      ; 
                        sta TabBulletEnemyDist      ; 
                        
.ChkMax                 lda #$b0                    ; 
                        cmp TabBulletEnemyDist      ; 
                        bcs .ExitBulletClr          ; greater/equal
                        
                        sta TabBulletEnemyDist      ; 
                        
.ExitBulletClr          rts                         ; 
                        
.BulletHitsShip         jsr ResetVoc1               ; 
                        
                        lda #$df                    ; ##.#####
                        and TabSfxToPlay            ; 
                        ora #$10                    ; ...#....
                        sta TabSfxToPlay            ; 
                        
                        lda #$06                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$0d                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        jsr ResetVoc3               ; 
                        
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        dec TabPlayerLives          ; 
                        lda TabPlayerLives          ; 
                        bne .Destroyed              ; 
                        
                        jsr ScreenStatusGameOver    ; 
                        
                        lda #TabShowScoresNo        ; 
                        sta TabPlayerShowScores     ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        sta TabP_ShowScores,x       ; 
                        
                        jsr SwitchPlayerAndValues   ; 
                        beq .DecPlayers             ; player #1
                        
                        dec TabNumberOfPlayers      ; player #2
                        
                        pla                         ; 
                        pla                         ; 
                        jsr ScreenStatusMap         ; 
                        jsr ClearStatusRows3_5      ; 
                        jsr ScreenStatusInitShip    ; ship shoot out
                        jmp Chapter_ShipShootOut    ; 
                        
.DecPlayers             dec TabNumberOfPlayers      ; 
                        beq .AllPlayersOut          ; 
                        
                        pla                         ; 
                        pla                         ; 
                        jmp InitBayArea             ; with RTS
                        
.AllPlayersOut          jmp ThreeLevel_RTS          ; Game Over
                        
.Destroyed              jsr ResetTOD1               ; 
                        
                        lda #TabSeaShipMsgDestroyYes; 
                        sta TabSeaShipMsgDestroy    ; 
                        
                        lda #$94                    ; 
                        sta TabBulletEnemyDist      ; 
                        jsr NextEnemyBulletPhase    ; 
                        
SeaShipEnemyBulletX     jmp .ClrBullet              ; 
; -------------------------------------------------------------------------------------------------------------- ;
; CannonPlayerBullet    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
CannonPlayerBullet      subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkBulletPase_00       ; TabSpriteStatusUsed
                        
.ExitNoShot             rts                         ; TabSpriteStatusFree
                        
.ChkBulletPase_00       lda #TabFlagTankEneShootNo  ; 
                        sta TabFlagTankEneShoot     ; 
                        
                        lda TabPlayerBulletPhaseNo  ; 
                        bne .ChkBulletPase_01       ; 
                        
                        lda #$0b                    ; 
                        sta __PlayerShootSprtPtrAdd ; 
                        
                        lda #BLACK                  ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        jsr CannonPlayerShoot       ; 
                        
                        lda #$09                    ; 
                        sta __PlayerShootSprtPtrAdd ; 
                        
                        lda #TabSpriteStatusUsed    ; 
                        sta TabP_ShotStatus         ; 
                        
.ExitPase_00            rts                         ; 
                        
.ChkBulletPase_01       cmp #$01                    ; 
                        bne .ChkBulletPase_02       ; 
                        
                        ldx #$01                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$04                    ; 
                        sta TabWaitTimeBullFlyPhSav ; 
                        sta TabWaitTimeBullFlyPh    ; 
                        sta TabWaitFlagBulletFlyPh  ; TabWaitFlag_Yes + extra
                        
                        inc TabPlayerBulletPhaseNo  ; 
                        
                        lda #$b6                    ; #.## .##.
                        sec                         ; 
                        sbc TabCannonShotDegree     ; 
                        sta TabCannonShotDist       ; 
                        
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        sta TabCannonShotHeight     ; 
                        
                        lda #$a2                    ; 
                        sec                         ; 
                        sbc TabCannonShotHeight     ; 
                        sta TabCannonShotHeight     ; 
                        
                        lda #$84                    ; 
                        sta TabMoveTorpedo_02       ; 
                        
                        lda #TabSpriteActiveYes     ; 
                        sta TabSpriteActiveBulletP1 ; 
                        
                        lda #$00                    ; 
                        sta TabMoveTorpedo_01       ; 
                        sta TabPlaneSpeed_Val_1     ; 
                        
.ExitPase_01            rts                         ; 
                        
.ChkBulletPase_02       cmp #$02                    ; 
                        bne .ExitPase_01            ; 
                        
                        lda TabWaitFlagBulletFlyPh  ; 
                        bne .ExitPase_01            ; TabWaitFlag_Yes
                        
                        inc TabMoveTorpedo_02       ; 
                        inc TabWaitFlagBulletFlyPh  ; TabWaitFlag_Yes
                        
                        lda SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        cmp TabCannonShotHeight     ; 
                        bcc .ExitPase_01            ; lower
                        
                        lda #TabSpriteActiveNo      ; 
                        sta TabSpriteActiveBulletP1 ; 
                        
                        ldx #$01                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        lda TabFlagCarrierActive    ; 
                        beq .ChkVesselMsbPosX       ; TabFlagCarrierActiveNo
                        
                        lda #$02                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .ChkVesselMsbPosX       ; 
                        
                        lda SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$0c                    ; 
                        bcc .GetSp7PosX             ; 
                        
                        cmp SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcs .ChkVesselMsbPosX       ; 
                        
.GetSp7PosX             lda SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc #$08                    ; 
                        cmp SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bcc .ChkVesselMsbPosX       ; lower
                        
                        lda TabCannonShotDist       ; 
                        cmp #$60                    ; 
                        bne .Sub60                  ; 
                        
                        jsr SeaShipSinkCarrier      ; 
                        
.ReInitShotAndExit      lda #$00                    ; 
                        sta TabP_ShotStatus         ; TabSpriteStatusFree
                        sta TabPlayerBulletPhaseNo  ; 
                        
                        rts                         ; 
                        
.Sub60                  sec                         ; 
                        sbc #$60                    ; 
                        bcc .MakePositive           ; underflow
                        
                        jsr ScreenStatusShowLong    ; shoot was too long
                        jmp .ReInitShotAndExit      ; 
                        
.MakePositive           eor #$ff                    ; make positive
                        clc                         ; 
                        adc #$01                    ; 
                        jsr ScreenStatusShowShort   ; shoot was too short
                        jsr BulletWaterSplash       ; 
                        jmp .ReInitShotAndExit      ; 
                        
.ChkVesselMsbPosX       lda #$02                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .ChkPosVessel_4         ; 
                        
.ChkPosVessel_1         lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$4c                    ; 
                        bcs .ChkPosVessel_2         ; greater/equal
                        
                        lda TabSeaVesselActive_01   ; 
                        beq .OutStatusDistance      ; TabSeaVesselActiveNo
                        
                        lda TabCannonShotDist       ; 
                        sec                         ; 
                        sbc #$80                    ; 
                        bne .OutStatusDistTooLong   ; 
                        
                        lda #$02                    ; ......#.
                        bit TabSprtBackCollReg      ; 
                        beq .OutStatusDistance      ; 
                        
                        jsr SinkVessel_1            ; 1st from left to right
                        jmp .ReInitShotAndExit      ; 
                        
.ChkPosVessel_2         cmp #$84                    ; 
                        bcs .ChkPosVessel_3         ; greater/equal
                        
                        lda TabSeaVesselActive_02   ; 
                        beq .OutStatusDistance      ; TabSeaVesselActiveNo
                        
                        lda TabCannonShotDist       ; 
                        sec                         ; 
                        sbc #$70                    ; 
                        bne .OutStatusDistTooLong   ; 
                        
                        lda #$02                    ; 
                        bit TabSprtBackCollReg      ; 
                        beq .OutStatusDistance      ; 
                        
                        jsr SinkVessel_2            ; 2nd from left to right
                        jmp .ReInitShotAndExit      ; 
                        
.ChkPosVessel_3         lda TabSeaVesselActive_03   ; 
                        beq .OutStatusDistance      ; TabSeaVesselActiveNo
                        
                        lda TabCannonShotDist       ; 
                        sec                         ; 
                        sbc #$8c                    ; 
                        bne .OutStatusDistTooLong   ; 
                        
                        lda #$02                    ; ......#.
                        bit TabSprtBackCollReg      ; 
                        beq .OutStatusDistance      ; 
                        
                        jsr SinkVessel_3            ; 3rd from left to right
                        jmp .ReInitShotAndExit      ; 
                        
.ChkPosVessel_4         lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$0c                    ; 
                        bcc .ChkPosVessel_3         ; 
                        
                        lda TabSeaVesselActive_04   ; 
                        beq .OutStatusDistance      ; TabSeaVesselActiveNo
                        
                        lda TabCannonShotDist       ; 
                        sec                         ; 
                        sbc #$28                    ; 
                        bne .OutStatusDistTooLong   ; 
                        
                        lda #$02                    ; 
                        bit TabSprtBackCollReg      ; 
                        beq .OutStatusDistance      ; 
                        
                        jsr SinkVessel_4            ; 4th from left to right
                        jmp .ReInitShotAndExit      ; 
                        
.OutStatusDistance      lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDist ; 
                        ldx #BH_Scrn_Stat_LenDist ; 
                        jsr ScreenStatusTextOutRow5 ; xr=lenght blank out  yr=output text offset
                        jmp .BulletHitsWater        ; 
                        
.OutStatusDistTooLong   bcc .OutStatusDistTooShort  ; 
                        jsr ScreenStatusShowLong    ; shoot was too long
                        
                        lda #$02                    ; ......#.
                        bit TabSprtBackCollReg      ; 
                        bne .GoInitShotAndExit      ; 
                        
.BulletHitsWater        jsr BulletWaterSplash       ; 
.GoInitShotAndExit      jmp .ReInitShotAndExit      ; 
                        
.OutStatusDistTooShort  eor #$ff                    ; 
                        clc                         ; 
                        adc #$01                    ; 
                        jsr ScreenStatusShowShort   ; shoot was too short
CannonPlayerBulletX     jmp .BulletHitsWater        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipSinkCarrier    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipSinkCarrier      subroutine                  ; 
                        ldx #TabTargetShipHitCarr   ; carrier
                        jsr SeaShipVesselExplode    ; 
                        
                        dec TabFlagCarrierActive    ; TabFlagCarrierActiveNo
                        
                        lda SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$9c                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Ship_35         ; cover block
                        sta BH_SpritePointer03_A    ; 
                        sta BH_SpritePointer04_A    ; 
                        
                        lda #$e7                    ; ###..###
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BLUE                   ; water
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
                        lda #$e7                    ; ###..###
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$18                    ; 
                        ora XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        ldx #$03                    ; ......##
                        jsr EnableSpriteNoFromXr    ; 
                        
                        ldx #$04                    ; .....#..
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$0f                    ; ....####
                        sta TabWaitSinkShipSav      ; 
                        sta TabWaitSinkShip         ; 
                        
                        lda #TabTargetShipHitCarr   ; 
                        sta TabTargetShipHit        ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagSinkShip     ; 
SeaShipSinkCarrierX     rts
; -------------------------------------------------------------------------------------------------------------- ;
; SinkVessel_1          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SinkVessel_1            subroutine                  ; 1st from left to right
                        lda #$38                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$8a                    ; 
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        lda #$9a                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Ship_36         ; vessel 1
                        sta BH_SpritePointer04_A    ; 
                        
                        jsr SeaShipDrownVessel      ; 
                        
                        lda #$73                    ; clear background - sky
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_1 + $00                   ; $49bc
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_1 + BH_Scrn_RowLen        ; $49e4
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_1 + BH_Scrn_RowLen + $01  ; $49e5
                        
                        ldx #$01                    ; 
                        stx TabTargetShipHit        ; 
                        jsr SeaShipVesselExplode    ; 
                        
                        lda #TabSeaVesselActiveNo   ; 
                        sta TabSeaVesselActive_01   ; 
                        
SinkVessel_1X           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipVesselScore    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipVesselScore      subroutine                  ; 
                        lda TabWaitFlagSinkShip     ; 
                        bne SeaShipVesselScoreX     ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSinkShip     ; TabWaitFlag_Yes
                        
                        inc SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
                        lda SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        cmp SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        bne SeaShipVesselScoreX     ; 
                        
                        lda #$e7                    ; ###..###
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
.Score                  jsr AddScore_2000           ; 
                        
                        lda #TabTargetShipHitNo     ; 
                        sta TabTargetShipHit        ; 
                        
SeaShipVesselScoreX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipDrownVessel    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipDrownVessel      subroutine                  ; 
                        lda #SP_Ptr_Ship_35         ; cover block
                        sta BH_SpritePointer03_A    ; 
                        
                        lda #$e7                    ; ###..###
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BLACK                  ; 
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
                        lda #BLUE                   ; water
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        
                        lda #$e7                    ; ###..###
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$e7                    ; ###..###
                        and XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #$18                    ; ...##...
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #$0f                    ; ....####
                        sta TabWaitSinkShipSav      ; 
                        sta TabWaitSinkShip         ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagSinkShip     ; 
                        
SeaShipDrownVesselX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SinkVessel_2          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SinkVessel_2            subroutine                  ; 2nd from left to right
                        lda #$70                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        lda #$8a                    ; 
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        lda #$9b                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Ship_37         ; vessel 2
                        sta BH_SpritePointer04_A    ; 
                        
                        jsr SeaShipDrownVessel      ; 
                        
                        lda #$73                    ; clear background - sky
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_2 + $00  ; $49eb
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_2 + $01  ; $49ec
                        lda #$30                    ; clear background - sea
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_2 + BH_Scrn_RowLen - $01  ; $4a12
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_2 + BH_Scrn_RowLen + $00  ; $4a13
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_2 + BH_Scrn_RowLen + $01  ; $4a14
                        
                        ldx #$02                    ; 
                        stx TabTargetShipHit        ; 
                        jsr SeaShipVesselExplode    ; 
                        
                        lda #TabSeaVesselActiveNo   ; 
                        sta TabSeaVesselActive_02   ; 
                        
SinkVessel_2X           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SinkVessel_3          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SinkVessel_3            subroutine                  ; 3rd from left to right
                        lda #$00                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$8a                    ; 
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
                        lda #$9a                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Ship_38         ; vessel 3
                        sta BH_SpritePointer04_A    ; 
                        
                        jsr SeaShipDrownVessel      ; 
                        
                        lda #$73                    ; clear background - sky
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_3 ; $49fe
                        
                        lda #$18                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        ldx #$03                    ; 
                        stx TabTargetShipHit        ; 
                        jsr SeaShipVesselExplode    ; 
                        
                        lda #TabSeaVesselActiveNo   ; 
                        sta TabSeaVesselActive_03   ; 
                        
SinkVessel_3X           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SinkVessel_4          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SinkVessel_4            subroutine                  ; 4th from left to right
                        lda #$30                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$92                    ; 
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
                        lda #$a0                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Ship_39         ; vessel 4
                        sta BH_SpritePointer04_A    ; 
                        
                        jsr SeaShipDrownVessel      ; 
                        
                        lda #$18                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$18                    ; 
                        ora XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #$30                    ; clear background - sea
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_4 + $00 ; $4a2b
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_4 + $01 ; $4a2c
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_4 + $02 ; $4a2d
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_4 + $03 ; $4a2e
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrVessel_4 + $04 ; $4a2f
                        
                        ldx #$04                    ; 
                        stx TabTargetShipHit        ; 
                        jsr SeaShipVesselExplode    ; 
                        
                        lda #TabSeaVesselActiveNo   ; 
                        sta TabSeaVesselActive_04   ; 
                        
SinkVessel_4X           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipVesselExplode  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipVesselExplode    subroutine                  ; 
                        lda #SP_Ptr_Ship_23         ; explosion
                        sta BH_SpritePointer01_A    ; 
                        
                        lda #TabSeaVesselActiveNo   ; 
                        sta TabSeaVesselActive,x    ; 
                        
                        jsr SetAcToXrTimesMult2     ; 
                        eor #$ff                    ; ########
                        and TabPlayerVesselActive   ; one bit per ship
                        sta TabPlayerVesselActive   ; 
                        
                        cpx TabEnemyBulletPhase     ; 
                        bne .Enable                 ; 
                        
                        jsr NextEnemyBulletPhase    ; 
                        
                        lda #$94                    ; 
                        sta TabBulletEnemyDist      ; 
                        
.Enable                 ldx #$01                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc #$0c                    ; 
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
                        lda #TabTargetHitYes        ; 
                        sta TabTargetHit            ; 
                        
                        lda #BH_SprtColorShipExplode; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        
                        lda #BH_SprtColorShipExplodeBkgr ; 
                        sta TabBackGroundColor      ; flash
                        
                        jsr ResetTOD1               ; 
                        
                        lda #$10                    ; ...#....
                        bit TabSfxToPlay            ; 
                        beq .SfxExplodeInit         ; 
                        
                        rts                         ; 
                        
.SfxExplodeInit         jsr ResetVoc1               ; 

                        lda #$50                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$0d                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
SeaShipVesselExplodeX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowLong  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowLong    subroutine                  ; 
                        jsr ScreenStatusShowDist    ; 
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDistLong ; 
                        ldx #<TabTextStatusMLong    ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
ScreenStatusShowLongX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowShort Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowShort   subroutine                  ; 
                        jsr ScreenStatusShowDist    ; 
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDistShort ; 
                        ldx #<TabTextStatusMShort   ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
ScreenStatusShowShortX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowDist  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowDist    subroutine                  ; 
                        jsr MakeDecimal             ; xr=100 yr=10_1
                        
                        tya                         ; 
                        pha                         ; 
                        txa                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffDist10000
                        
                        pla                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        stx BH_Scrn_Stat + BH_Scrn_Stat_OffDist1000
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffDist100
                        
                        lda #$30                    ; "0"
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffDist10
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffDist1
                        
                        lda #$20                    ; " "
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffDist_
                        
                        lda #WHITE                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffDist10000
                        sta COLORAM + BH_Scrn_Stat_OffDist1000
                        sta COLORAM + BH_Scrn_Stat_OffDist100
                        sta COLORAM + BH_Scrn_Stat_OffDist10
                        sta COLORAM + BH_Scrn_Stat_OffDist1
                        
ScreenStatusShowDistX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MakeDecimal           Does    : 
;                       Expects : 
;                       Returns : xr=100 yr=10_1
; -------------------------------------------------------------------------------------------------------------- ;
MakeDecimal             subroutine                  ; 
                        ldy #$00                    ; 
.Sub100                 sta TabWrkMakeDecimal       ; 
                        sec                         ; 
                        sbc #$64                    ; 
                        bcc .Set100                 ; lower 100 - care for 10
                        
                        iny                         ; 
                        jmp .Sub100                 ; 
                        
.Set100                 tya                         ; 
                        tax                         ; count 100
                        lda TabWrkMakeDecimal       ; 
                        
                        ldy #$00                    ; 
.Sub10                  sta TabWrkMakeDecimal       ; 
                        sec                         ; 
                        sbc #$0a                    ; 
                        bcc .Set10_1                ; 
                        
                        iny                         ; 
                        jmp .Sub10                  ; 
                        
.Set10_1                tya                         ; 
                        asl a                       ; shift 10 to left halfbyte
                        asl a                       ; 
                        asl a                       ; 
                        asl a                       ; 
                        clc                         ; 
                        adc TabWrkMakeDecimal       ; add 1
                        tay                         ; 
                        
MakeDecimalX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowDegree Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowDegree  subroutine                  ; 
                        lda #$b3                    ; 
                        sec                         ; 
                        sbc TabCannonShotDegree     ; 
                        clc                         ; 
                        lsr a                       ; shift rightmost bit out
                        pha                         ; save degrees
                        bcc .GetZero10th            ; 
                        
.GetFive10th            lda #$05                    ; 
                        jmp .Out10th                ; 
                        
.GetZero10th            lda #$00                    ; 
                        
.Out10th                jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffDeg_1
                        pla                         ; restore degrees
                        
                        clc                         ; 
                        adc TabShipCannonDegree     ; 
                        jsr MakeDecimal             ; xr=100 yr=10_1
                        
                        tya                         ; 
                        jsr MakeDigit               ; xr=right nibble yr=left nibble
                        
                        stx BH_Scrn_Stat + BH_Scrn_Stat_OffDeg10
                        sty BH_Scrn_Stat + BH_Scrn_Stat_OffDeg1
                        
                        lda #$2e                    ; "."
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffDeg_
                        
                        lda #WHITE                  ; 
                        sta COLORAM + BH_Scrn_Stat_OffDeg10
                        sta COLORAM + BH_Scrn_Stat_OffDeg1
                        sta COLORAM + BH_Scrn_Stat_OffDeg_
                        sta COLORAM + BH_Scrn_Stat_OffDeg_1
                        
                        lda #WHITE                  ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDegElev ; 
                        ldx #<TabTextStatusDegElev + $02 ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
ScreenStatusShowDegreeX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; BulletWaterSplash     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
BulletWaterSplash       subroutine                  ; 
                        lda TabTargetShipHit        ; 
                        bmi .ChkPhase               ; TabTargetShipHitNo
                        
                        rts                         ; 
                        
.ChkPhase               lda TabCannonShotDist       ; 
.ChkPhaseMax            cmp #$41                    ; 
                        bcs .ChkPhaseMid            ; higher/equal
                        
.GetSplashMax           lda #SP_Ptr_Ship_26         ; water splash phase 3 - max
                        jmp .SetSplashSpritePtr     ; 
                        
.ChkPhaseMid            cmp #$63                    ; 
                        bcs .GetSplashMin           ; higher/equal
                        
.GetSplashMid           lda #SP_Ptr_Ship_25         ; water splash phase 2
                        jmp .SetSplashSpritePtr     ; 
                        
.GetSplashMin           lda #SP_Ptr_Ship_24         ; water splash phase 1 - min
                        
.SetSplashSpritePtr     sta BH_SpritePointer03_A    ; 
                        
                        lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda TabCannonShotHeight     ; 
                        sec                         ; 
                        sbc #$0a                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #$02                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .SetPosX_Msb            ; 
                        
.ClrPosX_Msb            lda #$f7                    ; ####.###
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .SetSplashColor         ; 
                        
.SetPosX_Msb            lda #$08                    ; ....#...
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.SetSplashColor         lda #BH_SprtColorShipSplash ; 
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        
                        lda #$f7                    ; 
                        and YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        
                        lda #$f7                    ; 
                        and XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #TabBulletWaterSplashYes; 
                        sta TabBulletWaterSplash    ; 
                        
                        lda #$14                    ; 
                        sta TabWaitWaterSplash      ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagAutoAction   ; 
                        
                        ldx #$03                    ; ......##
                        jsr EnableSpriteNoFromXr    ; 
                        
BulletWaterSplashX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; BulletWaterSplashOff  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
BulletWaterSplashOff    subroutine                  ; 
                        lda TabBulletWaterSplash    ; 
                        bne .ChkWait                ; TabBulletWaterSplashYes
                        
.Exit                   rts                         ; 
                        
.ChkWait                lda TabWaitFlagAutoAction   ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        lda #TabBulletWaterSplashNo ; 
                        sta TabBulletWaterSplash    ; 
                        
                        ldx #$03                    ; ......##
                        jsr DisableSpriteNoFromXr   ; 
BulletWaterSplashOffX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ShipInitCarrier       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ShipInitCarrier         subroutine                  ; 
                        lda #$a8                    ; 
                        sta SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc #$18                    ; 
                        sta SP6X                    ; VIC($D00C) Sprite 6 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc #$18                    ; 
                        sta SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$8a                    ; 
                        sta SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        sta SP6Y                    ; VIC($D00D) Sprite 6 Y-Pos (Bits 0-7)
                        sta SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        
                        ldx #$1f                    ; ...#####
                        txa                         ; 
                        and YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        txa                         ; 
                        and XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        txa                         ; 
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        txa                         ; 
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #BLACK                  ; 
                        sta SP5COL                  ; VIC($D02C) Color Sprite 5
                        sta SP6COL                  ; VIC($D02D) Color Sprite 6
                        sta SP7COL                  ; VIC($D02E) Color Sprite 7
                        
                        ldx #SP_Ptr_Ship_32         ; carrier nose
                        stx BH_SpritePointer05_A    ; 
                        inx                         ; carrier mid
                        stx BH_SpritePointer06_A    ; 
                        inx                         ; carrier rear
                        stx BH_SpritePointer07_A    ; 
                        
                        lda #$e0                    ; ###.....
                        ora SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        ldy #$12                    ; 
                        ldx #$0a                    ; 
                        lda #$73                    ; sky
.ClrCarrierSky          sta BH_Scrn_Gfx + BH_Scrn_Row_0d,y ; now moving - painted for plane shootout
                        iny                         ; 
                        dex                         ; 
                        bne .ClrCarrierSky          ; 
                        
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffClrCarrier ; clear remaining carrier top
                        
                        ldy #$12                    ; 
                        ldx #$0a                    ; 
                        lda #$30                    ; water
.ClrCarrierWater        sta BH_Scrn_Gfx + BH_Scrn_Row_0e,y ; now moving - painted for plane shootout
                        iny                         ; 
                        dex                         ; 
                        bne .ClrCarrierWater        ; 
                        
                        lda #$0f                    ; 
                        sta TabWaitMoveCarrierSav   ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveShip     ; 
                        
ShipInitCarrierX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipMoveCarrier    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipMoveCarrier      subroutine                  ; 
                        lda TabFlagCarrierActive    ; 
                        bne .ChkMove                ; TabFlagCarrierActiveYes
                        
.Exit                   rts                         ; 
                        
.ChkMove                lda TabWaitFlagMoveShip     ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagMoveShip     ; TabWaitFlag_Yes
                        
.ChkNosePosX            lda SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .ChkMidPosX              ; 
                        
                        dec SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
.ChkMidPosX             lda SP6X                    ; VIC($D00C) Sprite 6 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .ChkRearPosX            ; 
                        
                        dec SP6X                    ; VIC($D00C) Sprite 6 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
.ChkRearPosX            lda SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .Deactiveate            ; 
                        
                        dec SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        rts                         ; 
                        
.Deactiveate            lda #$1f                    ; ...#####
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        dec TabFlagCarrierActive    ; TabFlagCarrierActiveNo
                        
                        lda #$df                    ; ##.#####
                        and TabPlayerVesselActive   ; one bit per ship
                        sta TabPlayerVesselActive   ; 
                        
SeaShipMoveCarrierX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipDemo           Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipDemo             subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .ChkDemoTime            ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.ChkDemoTime            lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$30                    ; 
                        bne .ContinueDemo           ; 
                        
                        jsr ResetTOD2               ; 
                        
                        lda #TabGameModeDemoToBeach ; fleet moves to beach
                        sta TabGameMode             ; 
                        
                        pla                         ; 
                        pla                         ; 
                        
                        lda #$04                    ; .....#..
                        sta TabPlayerLives          ; 
                        
.Init                   jsr AcToTabSpritesActive    ; 
                        jsr SwitchPlayerAndValues   ; 
                        jmp InitBayArea             ; with RTS
                        
.ContinueDemo           lda #$01                    ; .......#
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ChkCannonPosX          ; 
                        
.GetMovLeft             lda #CIA_Joy_Le             ; 
                        sta TabJoysticAction        ; 
                        jmp .ChkDistShot            ; 
                        
.ChkCannonPosX          lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$63                    ; 
                        beq .ChkDistShot            ; 
                        bcs .GetMovLeft             ; 
                        
.GetMovRight            lda #CIA_Joy_Ri             ; 
                        sta TabJoysticAction        ; 
                        
.ChkDistShot            lda TabCannonShotDist       ; 
                        cmp #$60                    ; 
                        beq .MovFire                ; 
                        bcc .GetMovDown             ; 
                        
.GetMovUp               lda #CIA_Joy_Up             ; 
                        
.SetMovUpDo             ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
.MovFire                lda #CIA_Joy_Fi             ; 
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
.Exit                   rts                         ; 
                        
.GetMovDown             lda #CIA_Joy_Do             ; 
SeaShipDemoX            bne .SetMovUpDo             ; always
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitShip  Does    : Ship schoot out
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitShip    subroutine                  ; 
                        lda #BH_StatColorAmmoShipTxt; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoShipTxt ; 
                        ldx #<TabTextStatus250mm    ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorAmmoShip   ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoShipCube ; 
                        ldx #<TabTextStatusAmmoCube ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorDegElev    ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffDegElevTxt ; 
                        ldx #<TabTextStatusDegElev  ; 
                        jsr ScreenStatusTextOutRow4 ; xr=input text offset  yr=output text offset
                        
                        lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapAerial ; 
                        ldx #BH_Scrn_Stat_LenMapAerial ; 
                        jsr ScreenStatusTextOutRow5 ; xr=length to clear yr=output text offset
                        
ScreenStatusInitShipX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapHitBeachTankHill   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapHitBeachTankHill     subroutine                  ; 
                        lda TabBayMapSeaChapters    ; 
                        beq .ChkBeachPosX           ; TabBayMapSeaChaptersNo
                        
.Exit                   rts                         ; 

.ChkBeachPosX           lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #BH_BeachMinPosX        ; 
                        bcc .Exit                   ; lower
                        
                        cmp #BH_BeachMaxPosX        ; 
                        bcs .Exit                   ; higher/equal
                        
.ChkBeachPosY           lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #BH_BeachMinPosY        ; 
                        bcc .Exit                   ; lower
                        
                        cmp #BH_BeachMaxPosY        ; 
                        bcs .Exit                   ; higher/equal
                        
                        ldx #$01                    ; max players
.CpyShip2TankCount      lda TabPlayerLivesSav,x     ; 
                        asl a                       ; 
                        sta TabPlayerTanksSav,x     ; 
                        dex                         ; 
                        bpl .CpyShip2TankCount      ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        jsr PlayerValuesRestore     ; 
                        
                        jsr ChapterStartBell        ; 
MapHitBeachTankHillX    jmp Chapter_TanksOnBeach    ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_TanksOnBeach  Does    : Chapter: Tanks on the beach
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_TanksOnBeach    subroutine                  ; 
                        jsr ScreenStatusSave        ; 
                        jsr ScreenStatusGetReady    ; 
                        jsr ScreenStatusRestore     ; 
                        
                        jsr TankInitAll             ; init
                        jsr TankInitScreenSprites   ; init playground/player sprites
                        
                        lda #$02                    ; 
                        sta TabWaitTimeBeachLeRiSav ; equalize scroll speed
                        
                        lda #$03                    ; 
                        sta TabWaitTimeBeachUpDoSav ; 
                        
                        lda #$03                    ; 
                        sta TabWaitTimeTOD_Sec      ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr TankInitSfx             ; 
                        
.TanksOnBeach_Loop      jsr TankScrollTerrain       ; scroll terrain forward
                        jsr TankPlayerCheckShoots   ; player: check start shooting
                        jsr TankPlayerCheckLimit    ; player: clear sprite
                        jsr GetSprtSprtCollReg      ; sprite/sprite collision
                        jsr TankPlayerMovTank       ; player: move tank
                        jsr TankCheckPosCannon      ; enemy: cannon
                        jsr TankCheckPosTank        ; enemy: tank
                        jsr TankCheckPosMachGun     ; enemy: machine gun
                        jsr TankCheckPosTurret      ; enemy: turret
                        jsr TankEnemyMovTank        ; enemy: move tank
                        jsr TankEnemyShoot          ; enemy: shoots
                        jsr TankEnemyMovShot        ; enemy: move shot
                        jsr TankPlayerShoot         ; player: shoots
                        jsr TankPlayerMovShot       ; player: move shot
                        jsr TankPlayerHitCheck      ; player: hit check
                        jsr TankEnemyHitCheck       ; enemy: hit check
                        
                        jsr TankPlayerChkCollBack   ; 
                        jsr TankPlayerChkCollSprt   ; 
                        
                        jsr TankEnemyClrHitScore    ; enemy: wipe out hit score 
                        jsr PlayerAmmoInc           ; 
                        jsr TankAdaptSfxWithPosY    ; 
                        jsr ScreenStatusAmmoTank    ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr ScreenStatusShowTanks   ; 
                        jsr CheckGamePause          ; on C= commodore key
                        jsr CheckGameFinished       ; idle time/death - 3 level return to BASIC via GameStart
                        
                        lda TabScrnTankBayScrollPos ; 
                        beq Chapter_TanksOnBeachX   ; end of the beach - next chapter: storm the hill
                        
                        jsr TankClearSpriteAtMaxX   ; 
                        jsr CheckGameIdleTime       ; 
                        jsr TankDemo                ; 
                        jsr TankAdaptSpritePosX     ; keep PosX despite beach scrolling
                        
                        jmp .TanksOnBeach_Loop      ; 
                        
Chapter_TanksOnBeachX   jmp Chapter_StormTheHill    ; to Chapter: Storm the hill
; -------------------------------------------------------------------------------------------------------------- ;
; TankInitAll           Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankInitAll             subroutine                  ; 
                        jsr GameInit                ; 
                        
                        lda #$0a                    ; 
                        sta TabPlayerAmmo           ; 
                        jsr SpritesReset            ; 
                        
                        lda #$00                    ; 
                        sta TabTankShotSprtNoSav    ; 
                        sta TabTankScoreSpriteNo    ; 
                        
                        lda #$04                    ; 
                        sta TabTankShotSpeed        ; 
                        
                        lda #$03                    ; 
                        sta TabTankShotSpeedEne     ; 
                        
                        lda TabBeachSpriteLevel     ; 
                        sta TabBeachSpriteLevelTmp  ; 
                        
                        lda #$1e                    ; 
                        sta TabPlaySfxVoc2_01       ; 
                        
TankInitAllX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankInitScreenSprites Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankInitScreenSprites   subroutine                  ; 
                        lda #VIC_Cols38             ; ####.###
                        and SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        sta SCROLX                  ; VIC($D016) VIC Control Register 2 (Horiz Fine Scroll)
                        
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        jsr ClearGfxScreen          ; 
                        
                        lda #$00                    ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        ldx #<ScreenDataBayShore    ; 
                        ldy #>ScreenDataBayShore    ; 
                        jsr ScreenGraphicCharSet    ; 
                        
                        ldx #<ScreenDataStrandBridge; 
                        ldy #>ScreenDataStrandBridge; 
                        jsr ScreenGraphicCharSet    ; 
                        
                        ldx #BH_ChapterTanksOnBeach ; 
                        jsr SpriteDataCopyInit      ; 
                        
                        lda #TabTankScrollAreaTabPtrStart ; screen_00
                        sta TabTankScrollAreaTabPtr ; to TabTankScrollAreaPtrLo/Hi table
                        
                        jsr TankScreenShowStart     ; 
                        jsr TankSetBgColors         ; 
                        jsr ScreenStatusInitTank    ; 
                        jsr ScreenStatusAmmoTank    ; 
                        jsr ScreenStatusShowTanks   ; 
                        jsr TankInitMyTanks         ; 
                        
                        lda #TabScrnTankBayScrollPosIni ; 
                        sta TabScrnTankBayScrollPos ; 
                        
                        lda #$ff                    ; 
                        sta TabScrnTankBayScrollDec ; 
                        
TankInitScreenSpritesX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearGfxScreen        Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearGfxScreen          subroutine                  ; 
                        lda #$00                    ; 
                        ldx #$07                    ; 
.ClrSpritePtrs          sta BH_SpritePointers_Chr,x ; 
                        dex                         ; 
                        bpl .ClrSpritePtrs          ; 
                        
.SetScreen              lda #$00                    ; screen B
                        sta TabTankActiveScreen     ; 
                        
                        ldx #BH_Scrn_Gfx_Rows - $01 ; number of rows
.GetGfxScreenRowPtr     jsr TankObstIniScrnOutPtr   ; 
                        
                        ldy #BH_Scrn_RowLen - 1     ; 
                        lda #$ff                    ; 
.ClrGfxScreen           sta (BLZ_ScrnRowPtrGfxOut),y; 
                        dey                         ; 
                        bpl .ClrGfxScreen           ; 
                        
                        dex                         ; 
                        bpl .GetGfxScreenRowPtr     ; 
                        
ClearGfxScreenX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankInitMyTanks       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankInitMyTanks         subroutine                  ; 
                        ldx TabPlayerTanksLeft      ; 
                        dex                         ; 
.GetTankSpritePtr       lda #SP_Ptr_Beach_01        ; my tank move left
                        sta BH_SpritePointers_A,x   ; 
                        sta BH_SpritePointers_B,x   ; 
                        
                        lda #BH_SprtColorBeachTank  ; 
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda #$dc                    ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        dex                         ; 
                        bpl .GetTankSpritePtr       ; 
                        
                        lda #$01                    ; 
                        sta SPMC0                   ; VIC($D025) Sprite Multicolor Register 0
                        
                        lda #$00                    ; 
                        sta SPMC1                   ; VIC($D026) Sprite Multicolor Register 1
                        
                        lda #$00                    ; 
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #$ff                    ; 
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$00                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        ldx TabPlayerTanksLeft      ; 
                        dex                         ; 
                        
                        lda #$b8                    ; 
.SetTankPosY            pha                         ; 
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        pla                         ; 
                        sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sec                         ; 
.SetTankNextPosY        sbc #$0a                    ; 
                        pha                         ; 
                        jsr EnableSpriteNoFromXr    ; 
                        pla                         ; 
                        
                        dex                         ; 
                        bpl .SetTankPosY            ; 
                        
                        lda #$04                    ; 
                        sta TabPlaySfxVoc2_00       ; 
                        sta TabPlaySfxVoc1_00       ; 
                        
                        jsr ResetVoc1               ; 
                        jsr ResetVoc2               ; 
                        
                        lda #$14                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$30                    ; 
                        sta SUREL1                  ; SID($D406) Oscillator 1 Sustain/Release
                        lda #$00                    ; 
                        sta FRELO2                  ; SID($D407) Oscillator 2 Freq Ctrl (low byte)
                        lda #$02                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        lda #$50                    ; 
                        sta SUREL2                  ; SID($D40D) Oscillator 2 Sustain/Release
                        lda #$21                    ; 
                        sta VCREG2                  ; SID($D40B) Oscillator 2 Control
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
                        lda #$01                    ; 
                        sta TabPlaySfxVoc1_05       ; 
                        sta TabPlaySfxVoc2_05       ; 
                        
TankInitMyTanksX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusShowTanks Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusShowTanks   subroutine                  ; 
                        ldx #$00                    ; 
                        ldy TabPlayerTanksLeft      ; 
                        beq .ChkMax                 ; 
                        
.GetIndicator           lda #$71                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffTanksO,x ; 
                        
                        lda #BH_StatColorTanks      ; 
                        sta COLORAM + BH_Scrn_Stat_OffTanksO,x ; 
                        
                        inx                         ; 
.Gap                    lda #$20                    ; " "
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffTanksO,x ; 
                        inx                         ; 
                        dey                         ; 
                        bne .GetIndicator           ; 
                        
.ChkMax                 cpx #BH_TanksMax * $02      ; include gaps
                        bne .BlankOut               ; 
                        
.Exit                   rts                         ; 
                        
.BlankOut               lda #$20                    ; 
                        sta BH_Scrn_Stat + BH_Scrn_Stat_OffTanksO,x ; 
                        inx                         ; 
                        jmp .ChkMax                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerCheckLimit  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerCheckLimit    subroutine                  ; 
                        lda TabScrnTankBayScrollPos ; 
                        cmp #TabScrnTankBayScrollPosGo ; 
                        beq .SwitchTanksOff         ; 
                        
                        rts                         ; 
                        
.SwitchTanksOff         lda #$01                    ; .......# - leave player only
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
TankPlayerCheckLimitX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerMovTank     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerMovTank       subroutine                  ; 
                        lda TabWaitFlagMoveUpDo     ; 
                        bne .Exit_1                 ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagMoveUpDo     ; TabWaitFlag_Yes
                        
.ChkUp                  lda #CIA_Joy_Up             ; 
                        bit TabJoysticAction        ; 
                        beq .ChkDown                ; 
                        
                        jsr ClearDirInJoyAction     ; 
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$68                    ; 
                        bcc .Exit_1                 ; 
                        
                        dec SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #SP_Ptr_Beach_02        ; my tank move up
                        sta BH_SpritePointer00_A    ; 
                        sta BH_SpritePointer00_B    ; 
                        
.Exit_1                 rts                         ; 
                        
.ChkDown                lda #CIA_Joy_Do             ; 
                        bit TabJoysticAction        ; 
                        beq .TankSpriteDown         ; 
                        
                        jsr ClearDirInJoyAction     ; 
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp #$b8                    ; 
                        bcs .Exit_1                 ; 
                        
                        inc SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
.TankSpriteUp           lda #SP_Ptr_Beach_03        ; my tank move down
                        sta BH_SpritePointer00_A    ; 
                        sta BH_SpritePointer00_B    ; 
                        
                        rts                         ; 
                        
.TankSpriteDown         lda #SP_Ptr_Beach_01        ; my tank move left
                        sta BH_SpritePointer00_A    ; 
                        sta BH_SpritePointer00_B    ; 
                        
TankPlayerMovTankX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankCheckPosCannon    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankCheckPosCannon      subroutine                  ; 
                        ldx #$00                    ; 
.GetNextCannonPos       lda TabPosEnemyCannon,x     ; enemy: cannon
                        bne .ChkScrollPos           ; 
                        
                        rts                         ; 
                        
.ChkScrollPos           cmp TabScrnTankBayScrollPos ; 
                        beq .IniCannon              ; 
                        
                        inx                         ; 
                        inx                         ; 
                        jmp .GetNextCannonPos       ; 
                        
.IniCannon              inx                         ; 
                        lda TabPosEnemyCannon,x     ; enemy: cannon
                        ldy #BH_SprtColorBeachCannon; sprite color
                        ldx #$03                    ; sprite number
                        jsr TankInitEnemies         ; 
                        
TankCheckPosCannonX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankCheckPosTank      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankCheckPosTank        subroutine                  ; 
                        ldx #$00                    ; 
.GetNextEneTankPos      lda TabPosEnemyTank,x       ; enemy: tank
                        bne .ChkScrollPos           ; 
                        
                        rts                         ; 
                        
.ChkScrollPos           cmp TabScrnTankBayScrollPos ; 
                        beq .IniEneTank             ; 
                        
                        inx                         ; 
                        inx                         ; 
                        jmp .GetNextEneTankPos      ; 
                        
.IniEneTank             inx                         ; 
                        lda TabPosEnemyTank,x       ; enemy: tank
                        ldy #BH_SprtColorBeachTankE ; sprite color
                        ldx #$04                    ; sprite number
                        jsr TankInitEnemies         ; 
                        
TankCheckPosTankX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankCheckPosMachGun   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankCheckPosMachGun     subroutine                  ; 
                        ldx #$00                    ; 
.GetNextMacGunPosY      lda TabPosEnemyGunner,x     ; enemy: machine gun
                        bne .ChkScrollPos           ; 
                        
                        rts                         ; 
                        
.ChkScrollPos           cmp TabScrnTankBayScrollPos ; 
                        beq .IniMaxGun              ; 
                        
                        inx                         ; 
                        inx                         ; 
                        jmp .GetNextMacGunPosY      ; 
                        
.IniMaxGun                   inx                         ; 
                        lda TabPosEnemyGunner,x     ; enemy: machine gun
                        ldy #BH_SprtColorBeachMacGun; sprite color
                        ldx #$05                    ; sprite number
                        jsr TankInitEnemies         ; 
                        
TankCheckPosMachGunX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankCheckPosTurret    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankCheckPosTurret      subroutine                  ; 
                        ldx #$00                    ; 
.GetNextTurretPosY      lda TabPosEnemyTurret,x     ; enemy: turret
                        bne .ChkScrollPos           ; 
                        
                        rts                         ; 
                        
.ChkScrollPos           cmp TabScrnTankBayScrollPos ; 
                        beq .IniTurret              ; 
                        
                        inx                         ; 
                        inx                         ; 
                        jmp .GetNextTurretPosY      ; 
                        
.IniTurret              inx                         ; 
                        lda TabPosEnemyTurret,x     ; enemy: turret
                        ldy #BH_SprtColorBeachTurret; sprite color
                        ldx #$06                    ; sprite number
                        jsr TankInitEnemies         ; 
                        
TankCheckPosTurretX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankInitEnemies       Does    : 
;                       Expects : enemy sprite: ac=posy yr=color xr=number
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankInitEnemies         subroutine                  ; 
                        pha                         ; save PosY
                        
                        lda TabWaitFlagCaveShipUpDo ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagCaveShipUpDo ; TabWaitFlag_Yes
                        
                        lda TabBeachSpriteLevelTmp  ; 
                        jmp .ChkLevel               ; 
                        
.Exit                   pla                         ; restore PosY
                        rts                         ; 
                        
.ChkLevel               beq .GetColor               ; 
                        
                        dec TabBeachSpriteLevelTmp  ; 
                        pla                         ; restore PosY
                        
                        rts                         ; 
                        
.GetColor               tya                         ; 
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        
.GetPosY                pla                         ; restore PosY
                        sta SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        
                        lda #$00                    ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        txa                         ; 
                        clc                         ; 
                        adc #SP_Ptr_Beach_03        ; my tank move down
                        sta BH_SpritePointers_A,x   ; 
                        sta BH_SpritePointers_B,x   ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        ora SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        jsr SetAcToXrTimesMult2     ; 
                        
                        eor #$ff                    ; ########
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #TabSpriteStatusUsed    ; 
                        sta TabSpriteStatus,x       ; 
                        
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda TabBeachSpriteLevel     ; 
                        sta TabBeachSpriteLevelTmp  ; 
                        
TankInitEnemiesX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankEnemyMovTank      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankEnemyMovTank        subroutine                  ; 
                        lda TabFlagTankEneMove      ; 
                        bne .CopyPlayerPosY         ; 
                        
                        rts                         ; 
                        
.CopyPlayerPosY         lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
TankEnemyMovTankX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerCheckShoots Does    : .honz
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerCheckShoots   subroutine                  ; 
                        lda #$01                    ; .......#
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .ChkMinPosX             ; 
                        
.ChkWaitTime            lda TabWaitFlagCaveShipLeRi ; 
                        beq .SetWaitTime            ; TabWaitFlag_No
                        
.Exit                   rts                         ; 
                        
.SetWaitTime            inc TabWaitFlagCaveShipLeRi ; TabWaitFlag_Yes
                        inc SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        bne .Exit                   ; 
                        
                        lda #$01                    ; .......#
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        rts                         ; 
                        
.ChkMinPosX             lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$20                    ; 
                        bcc .ChkWaitTime            ; lower
                        
TankPlayerCheckShootsX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankEnemyShoot        Does    : .honz
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankEnemyShoot          subroutine                  ; 
                        lda TabWaitTankEneShoot     ; 
                        beq .ChkActiveShot          ; TabWaitTankEneShootNo
                        
                        rts                         ; 
                        
.ChkActiveShot          lda TabTankShotSprtNoSav    ; 
                        beq .IniNextSprite          ; inactive
                        
                        tax                         ; active
                        jmp .IniEneTankShot         ; 
                        
.IniNextSprite          ldx #$03                    ; 
.GetNextSprite          lda TabSpriteStatus,x       ; 
                        beq .SetNextSprite          ; 
                        
                        jsr SetAcToXrTimesMult2     ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .SetNextSprite          ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        
.GetPosY                lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
__TankEnemyShoot        equ * + $01                 ; 
                        cmp #$40                    ; min
                        bcc .SetNextSprite          ; lower
                        
                        cmp #$d8                    ; max
                        bcc .IniEneTankShot         ; lower
                        
.SetNextSprite          inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextSprite          ; 
                        
                        rts                         ; 
                        
.IniEneTankShot         lda #SP_Ptr_Beach_05        ; enemy tank
                        sta BH_SpritePointer02_A    ; 
                        sta BH_SpritePointer02_B    ; 
                        
                        lda #$fb                    ; #####.##
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$fb                    ; #####.##
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #BH_SprtColorBeachTankShotE ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc #$06                    ; 
                        sta SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
                        lda #TabFlagTankEneShootYes ; 
                        sta TabFlagTankEneShoot     ; 
                        sta TabWaitTankEneShoot     ; TabWaitTankEneShootYes
                        
.SetActiveShot          stx TabTankShotSprtNoSav    ; 
                        
                        ldx #$02                    ; ......#.
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeHillShotSavE ; 
                        sta TabWaitTimeHillShotE    ; 
                        
                        jsr TankSfxShot             ; 
TankEnemyShootX         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankEnemyMovShot      Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankEnemyMovShot        subroutine                  ; 
                        lda TabFlagTankEneShoot     ; 
                        bne .ChkWaitTime            ; TabFlagTankEneShootYes
                        
.Exit                   rts                         ; 
                        
.ChkWaitTime            lda TabWaitFlagAutoAction   ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagAutoAction   ; 
                        
                        lda #$04                    ; .....#..
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .AdaptShotPosX          ; 
                        
                        lda SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$58                    ; 
                        bcc .AdaptShotPosX          ; 
                        
.ClrShot                lda #$00                    ; 
                        sta TabFlagTankEneShoot     ; TabFlagTankEneShootNo
                        sta TabWaitTankEneShoot     ; TabWaitTankEneShootNo
                        
                        ldx #$02                    ; ......#.
                        jsr DisableSpriteNoFromXr   ; 
                        
                        rts                         ; 
                        
.AdaptShotPosX          lda TabTankAdaptPosX_02     ; 
                        clc                         ; 
                        adc TabTankShotSpeedEne     ; 
                        sta TabTankAdaptPosX_02     ; 
                        
TankEnemyMovShotX       rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerShoot       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerShoot         subroutine                  ; 
                        lda #CIA_Joy_Fi             ; 
                        bit TabJoysticAction        ; 
                        bne .ClrShot                ; 
                        
.Exit                   rts                         ; 

.ClrShot                jsr ClearDirInJoyAction     ; 
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$20                    ; 
                        bne .Exit                   ; 
                        
.ChkShotStatus          lda TabP_ShotStatus         ; 
                        bne .Exit                   ; TabSpriteStatusUsed
                        
                        lda #SP_Ptr_Beach_04        ; shot left
                        sta BH_SpritePointer01_A    ; 
                        sta BH_SpritePointer01_B    ; 
                        
                        lda #$fd                    ; ######.#
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BH_SprtColorBeachTankShot ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sec                         ; 
                        sbc #$06                    ; 
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
                        lda #$02                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        ldx #$01                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeTankShotSav  ; 
                        sta TabWaitTimeTankShot     ; 
                        sta TabP_ShotStatus         ; TabSpriteStatusUsed
                        
                        jsr TankSfxShot             ; 
                        jsr PlayerAmmoDec           ; 
                        
TankPlayerShootX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerMovShot     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerMovShot       subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkWaitTime            ; TabSpriteStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkWaitTime            lda TabWaitFlagMoveEnemy    ; 
                        bne .Exit                   ; 
                        
                        inc TabWaitFlagMoveEnemy    ; TabWaitFlag_Yes
                        
                        lda #$02                    ; ......#.
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        bne .MoveShotForward        ; 
                        
                        lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$30                    ; 
                        bcs .MoveShotForward        ; 
                        
                        lda #TabSpriteStatusFree    ; 
                        sta TabP_ShotStatus         ; 
                        
                        ldx #$01                    ; 
                        jsr DisableSpriteNoFromXr   ; 
                        
                        rts                         ; 
                        
.MoveShotForward        lda TabTankAdaptPosX_01     ; 
                        sec                         ; 
                        sbc TabTankShotSpeed        ; 
                        sta TabTankAdaptPosX_01     ; 
                        
TankPlayerMovShotX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerHitCheck    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerHitCheck      subroutine                  ; 
                        lda TabFlagTankEneShoot     ; 
                        bne .ChkHit                 ; TabFlagTankEneShootYes
                        
.Exit                   rts                         ; 
                        
.ChkHit                 lda #$05                    ; .....#.#
                        and TabSprtSprtCollReg      ; 
                        cmp #$05                    ; 
                        bne .Exit                   ; 
                        
                        ldx #$02                    ; ......#.
                        jsr DisableSpriteNoFromXr   ; 
; ------------------------------------------------- ;
TankPlayerHit           subroutine                  ; 
                        jsr TankSfxHit              ; 
                        jsr TankPlayerExplode       ; 
; ------------------------------------------------- ;
TankCheckLives          subroutine                  ; 
                        dec TabPlayerTanksLeft      ; 
                        beq .StatusGameOverPlayer_  ; 
                        
                        lda TabGameMode             ; 
                        bne .WasDemo                ; TabGameModeDemo or TabGameModeExit
                        
                        inc TabWrkNoUse             ; 
                        jsr SwitchPlayerAndValues   ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeBeachLeRiSav ; 
                        
                        pla                         ; 
                        pla                         ; 
                        jmp Chapter_TanksOnBeach    ; to Chapter: Tanks on the beach
                        
.WasDemo                jsr ResetTOD2               ; 
                        
                        pla                         ; level1
                        pla                         ; 
                        jmp Chapter_StormTheHill    ; to Chapter: Storm the hill
                        
.StatusGameOverPlayer_  jsr ScreenStatusGameOver    ; 
                        
                        lda #TabShowScoresNo        ; 
; ------------------------------------------------- ;
TankCheckGameOver       subroutine                  ; 
                        sta TabPlayerShowScores     ; 
                        
                        ldx TabActualPlayerNumber   ; 
                        lda #$00                    ; 
                        sta TabP_ShowScores,x       ; 
                        jsr SwitchPlayerAndValues   ; 
                        
                        dec TabNumberOfPlayers      ; 
.GameOver               beq TankPlayerHitCheckX     ; 
                        
.Continue               pla                         ; 
                        pla                         ; 
                        jsr ScreenStatusMap         ; 
                        jsr ScreenStatusInitTank    ; 
                        
                        jmp Chapter_TanksOnBeach    ; 
                        
TankPlayerHitCheckX     jmp ThreeLevel_RTS          ; Game Over
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerExplode     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerExplode       subroutine                  ; 
                        lda #SP_Ptr_Beach_00        ; explosion
                        sta BH_SpritePointer00_A    ; 
                        sta BH_SpritePointer00_B    ; 
                        
                        lda #$fe                    ; #######.
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #BH_SprtColorTankExplode; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #$06                    ; 
                        sta TabWaitTimeExplTankSav  ; 
                        sta TabWaitTimeExplTank     ; 
                        
.WaitIni                ldx #$09                    ; 
.Wait                   lda TabWaitFlagExplTank     ; 
                        bne .Wait                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagExplTank     ; TabWaitFlag_Yes
                        
                        dec SP0COL                  ; VIC($D027) Color Sprite 0
                        bpl .Wait                   ; 
                        
                        dex                         ; 
                        bne .Wait                   ; 
                        
TankPlayerExplodeX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankEnemyHitCheck     Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankEnemyHitCheck       subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkHit                 ; TabSpriteStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkHit                 lda #$02                    ; ......#.
                        bit TabSprtSprtCollReg      ; 
                        beq .Exit                   ; 
                        
                        ldx #$03                    ; 
.GetNextSpriteOff       jsr SetAcToXrTimesMult2     ; 
                        bit TabSprtSprtCollReg      ; 
                        bne .ShotStatusReset        ; 
                        
                        inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextSpriteOff       ; 
                        
                        rts                         ; 
                        
.ShotStatusReset        dec TabP_ShotStatus         ; TabSpriteStatusFree
                        
                        jsr TankSfxHit              ; 
                        
.ShotOff                lda #$fd                    ; ######.#
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        cpx TabTankShotSprtNoSav    ; 
                        bne .GetSpriteOff           ; 
                        
                        lda #$00                    ; 
                        sta TabTankShotSprtNoSav    ; 
                        
.GetSpriteOff           jsr SetAcToXrTimesMult2     ; 
                        eor #$ff                    ; ########
                        and SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$00                    ; BLACK/TabSpriteStatusFree
                        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        sta TabSpriteStatus,x       ; 
                        
                        lda #$3c                    ; 
                        sta TabWaitTimeBellsSav     ; 
                        sta TabWaitTimeBells        ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagPing         ; 
                        
.ScoreHits              stx TabTankScoreSpriteNo    ; 
                        
.ChkSpriteCannon        cpx #$03                    ; 
                        bne .ChkSpriteEneTank       ; 
                        
.ScoreSpriteCannon      jsr AddScore_800            ; 
                        lda #SP_Ptr_Beach_0d        ; score  800
                        jmp .SetScoreSpritePtrs     ; 
                        
.ChkSpriteEneTank       cpx #$04                    ; 
                        bne .ChkSpriteMacGun        ; 
                        
.ScoreSpriteEneTank     jsr AddScore_1000           ; 
                        lda #SP_Ptr_Beach_0e        ; score 1000
                        jmp .SetScoreSpritePtrs     ; 
                        
.ChkSpriteMacGun        cpx #$05                    ; 
                        bne .ScoreSpriteTurret      ; 
                        
.ScoreSpriteMacGun      jsr AddScore_400            ; 
                        lda #SP_Ptr_Beach_0b        ; score  400
                        jmp .SetScoreSpritePtrs     ; 
                        
.ScoreSpriteTurret      jsr AddScore_600            ; 
                        lda #SP_Ptr_Beach_0c        ; score  600
                        
.SetScoreSpritePtrs     sta BH_SpritePointers_A,x   ; 
                        sta BH_SpritePointers_B,x   ; 
                        
TankEnemyHitCheckX      rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankEnemyClrHitScore  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankEnemyClrHitScore    subroutine                  ; 
                        lda TabTankScoreSpriteNo    ; 
                        bne .ChkWait                ; 
                        
.Exit                   rts                         ; 
                        
.ChkWait                lda TabWaitFlagPing         ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        ldx TabTankScoreSpriteNo    ; 
                        
                        lda #$00                    ; 
                        sta TabTankScoreSpriteNo    ; 
                        
                        jsr DisableSpriteNoFromXr   ; 
TankEnemyClrHitScoreX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerChkCollBack Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerChkCollBack   subroutine                  ; 
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        and #$01                    ; .......#
                        bne TankPlayerChkCollBackX  ; 
                        
                        rts                         ; 
                        
TankPlayerChkCollBackX  jmp TankPlayerHit           ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankPlayerChkCollSprt Does    : .honz
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankPlayerChkCollSprt   subroutine                  ; 
                        lda #$01                    ; 
                        bit TabSprtSprtCollReg      ; 
                        bne .IniNextSpriteGet       ; 
                        
                        rts                         ; 
                        
.IniNextSpriteGet       ldx #$03                    ; ......##
.GetNextSpriteOff       jsr SetAcToXrTimesMult2     ; 
                        bit TabSprtSprtCollReg      ; 
                        bne TankPlayerChkCollSprtX  ; collision
                        
                        inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextSpriteOff       ; 
                        
                        rts                         ; 
                        
TankPlayerChkCollSprtX  jmp TankPlayerHit           ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankDemo              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankDemo                subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .ChkDemoTime            ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.ChkDemoTime            lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$25                    ; 
                        bne .ChkDemoTimeMov_Do      ; 
                        
                        jsr ResetTOD2               ; 
                        
                        pla                         ; 
                        pla                         ; 
                        jmp Chapter_StormTheHill    ; to Chapter: Storm the hill
                        
.ChkDemoTimeMov_Do      cmp #$10                    ; 
                        bne .ChkDemoTimeMov_Up      ; 
                        
                        lda #CIA_Joy_Do             ; 
                        sta TabJoysticAction        ; 
                        jmp .ChkDemoTimeMov_Fi       ; 
                        
.ChkDemoTimeMov_Up      cmp #$18                    ; 
                        bne .ChkDemoTimeMov_Fi      ; 
                        
                        lda #CIA_Joy_Up             ; 
                        sta TabJoysticAction        ; 
                        
.ChkDemoTimeMov_Fi      lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$02                    ; 
                        bcc TankDemoX               ; lower
                        
                        lda #CIA_Joy_Fi             ; 
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
TankDemoX               rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusInitTank  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusInitTank    subroutine                  ; 
                        lda #BH_StatColorAmmoTankTxt; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoTankTxt ; 
                        ldx #<TabTextStatus104mm    ; 
                        jsr ScreenStatusTextOutRow2 ; xr=input text offset  yr=output text offset
                        
.ClrOldEntry            lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffAmmoShipTxt ; 
                        ldx #BH_Scrn_Stat_LenAmmoShipTxt + BH_Scrn_Stat_LenAmmoShipCube ; 
                        jsr ScreenStatusTextOutRow3 ; xr=length blank out  yr=output text offset
                        
                        ldy #BH_Scrn_Stat_OffMapAerial ; 
                        ldx #BH_Scrn_Stat_LenMapAerial ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
                        lda #BH_StatColorTankText   ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffTanksTxt ; 
                        ldx #<TabTextStatusTanks    ; 
                        jsr ScreenStatusTextOutRow5 ; xr=input text offset  yr=output text offset
                        
ScreenStatusInitTankX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankSetBgColors       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankSetBgColors         subroutine                  ; 
                        lda #GREEN                  ; 
                        sta BGCOL2                  ; VIC($D023) Background Color 2
                        
                        lda #RED                    ; 
                        sta BGCOL1                  ; VIC($D022) Background Color 1
                        
                        lda #LT_GREY                ; 
                        sta TabBackGroundColor      ; 
                        
TankSetBgColorsX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Chapter_StormTheHill  Does    : Chapter: Storm the hill
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
Chapter_StormTheHill    subroutine                  ; 
                        jsr GameInit                ; 
                        
                        lda #$00                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #SP_Ptr_Ship_01         ; cannon shot phase  1 - max
                        sta TabPlrShootSprtBasePtr2 ; 
                        
                        lda #BLACK                  ; 
                        sta SP0COL                  ; VIC($D027) Color Sprite 0
                        
                        lda #$01                    ; .......#
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda TabVicMemControlSetA    ; 
                        sta TabVicMemControl        ; 
                        
                        lda #$01                    ; screen A
                        sta TabTankActiveScreen     ; 
                        
                        ldx #<ScreenDataHill        ; 
                        ldy #>ScreenDataHill        ; 
                        jsr ScreenGraphicInit       ; Init game screen data - Hill
                        
                        ldx #BH_ChapterStormTheHill ; 
                        jsr SpriteDataCopyInit      ; 
                        jsr HillCannonAutoPaint     ; paint hill cannon
                        jsr CannonSpritePosInit     ; 
                        jsr ClearGfxRow_20          ; 
                        
                        lda #$b5                    ; 
                        sta SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        jsr HillCannonNextTarget    ; set next hill target
                        
.StormTheHill_Loop      jsr GetSprtSprtCollReg      ; sprite/sprite collision
                        jsr GetSprtBackCollReg      ; sprite/foregr collision
                        jsr HillCannonAutoTarget    ; target hill cannon to player
                        jsr HillCannonAutoFire      ; fire   hill cannon
                        jsr CannonPlayerMove        ; move player cannon
                        jsr HillCannonPlayerFire    ; fire player cannon
                        jsr HillCannonPlayerShotFly ; 
                        jsr HillCannonCheckHits     ; check target hits
                        jsr HillCannonPlayerShotExp ; target hit explode
                        jsr PlayerAmmoInc           ; 
                        jsr ScreenStatusShowScore   ; 
                        jsr ScreenStatusShowDegree  ; 
                        jsr ScreenStatusAmmoTank    ; 
                        jsr ScreenStatusShowTanks   ; 
                        jsr HillCannonPlayerExpClr  ; 
                        jsr CheckGamePause          ; on C= commodore key
                        jsr CheckGameFinished       ; idle time/death - 3 level return to BASIC via GameStart
                        jsr CheckGameIdleTime       ; 
                        jsr HillDemo                ; 
                        
Chapter_StormTheHillX   jmp .StormTheHill_Loop      ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonAutoPaint   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonAutoPaint     subroutine                  ; 
                        lda #SP_Ptr_Hill_11         ; enemy cannon right step  1 - max left
                        sta BH_SpritePointer03_A    ; 
                        
                        lda #$90                    ; 
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$42                    ; 
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        
                        lda #$08                    ; 
                        ora XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        
                        lda #$00                    ; 
                        sta SP3COL                  ; VIC($D02A) Color Sprite 3
                        
                        lda #$10                    ; 
                        sta TabWaitBulletFlySav     ; 
                        
__HillCannonBulletFly   equ * + $01                 ; 
                        lda #$ff                    ; 
                        sta TabWaitBulletFly        ; 
                        
                        lda #TabPlaneDist_Ini       ; 
                        sta TabPlaneDistance        ; 
                        
                        lda #$01                    ; 
                        sta TabBulletStatus         ; TabBulletStatusUsed
                        sta TabWaitFlagBulletFly    ; TabWaitFlag_Yes
                        
                        ldx #$03                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        jsr ResetVoc1               ; 
                        lda #$02                    ; 
                        sta PWHI1                   ; SID($D403) Oscillator 1 Pulse Waveform Width (high nybble)
                        lda #$40                    ; 
                        sta SUREL1                  ; SID($D406) Oscillator 1 Sustain/Release
                        lda #$41                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        
HillCannonAutoPaintX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonNextTarget  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonNextTarget    subroutine                  ; 
                        ldx TabPlayerHillTargsToHit ; 
                        dex                         ; 
                        
                        lda TabHillTargetsPosX,x    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda TabHillTargetsPosY,x    ; 
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        
                        lda #$01                    ; 
                        sta SP4COL                  ; VIC($D02B) Color Sprite 4
                        
                        lda #SP_Ptr_Hill_27         ; hill target
                        sta BH_SpritePointer04_A    ; 
                        
                        ldx #$04                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
HillCannonNextTargetX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabHillTargetsPosX      dc.b $b9 ; 
                        dc.b $81 ; 
                        dc.b $a9 ; 
                        dc.b $d7 ; 
                        dc.b $79 ; 
                        dc.b $d9 ; 
                        dc.b $c7 ; 
                        dc.b $91 ; 
                        dc.b $c7 ; 
                        dc.b $a1 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabHillTargetsPosY      dc.b $6a ; 
                        dc.b $82 ; 
                        dc.b $5a ; 
                        dc.b $65 ; 
                        dc.b $75 ; 
                        dc.b $85 ; 
                        dc.b $5d ; 
                        dc.b $65 ; 
                        dc.b $7a ; 
                        dc.b $7d ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonAutoTarget  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonAutoTarget    subroutine                  ; 
                        lda $971b                   ; 
                        bne .ChkBulletWaitTime      ; 
                        
.Exit                   rts                         ; 
                        
.ChkBulletWaitTime      lda TabWaitFlagBulletFly    ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagBulletFly    ; TabWaitFlag_Yes
                        
                        lda #$c8                    ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        lda #$02                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        
                        inc BH_SpritePointer03_A    ; 
                        
                        lda BH_SpritePointer03_A    ; 
                        cmp #SP_Ptr_Hill_1e         ; enemy cannon down step 1 - up max
                        bne .ChkHillTarget          ; 
                        
                        lda TabPlaneDistance        ; 
                        bne .Exit                   ; 
                        
                        inc TabPlaneDistance        ; 
                        dec BH_SpritePointer03_A    ; 
                        
                        lda #$00                    ; 
                        sta FRELO1                  ; SID($D400) Oscillator 1 Freq Ctrl (low byte)
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        
                        lda #$28                    ; 
                        sta TabWaitBulletFly        ; 
                        
                        rts                         ; 
                        
.ChkHillTarget          cmp #SP_Ptr_Hill_27         ; hill target
                        bne .Exit                   ; 
                        
                        dec BH_SpritePointer03_A    ; 
                        
                        lda #$b0                    ; 
                        sta SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$43                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        
                        lda #BLACK                  ; 
                        sta SP2COL                  ; VIC($D029) Color Sprite 2
                        
                        lda #TabSpriteStatusFree    ; 
                        sta TabBulletStatus         ; 
                        
                        lda #$01                    ; 
                        sta TabFlagTankEneShoot     ; TabFlagTankEneShootYs
                        sta TabWaitFlagAutoAction   ; TabWaitFlag_Yes
                        
                        lda #$02                    ; 
                        sta TabWaitHillCannonAimSav ; 
                        sta TabWaitHillCannonAim    ; 
                        
                        jsr ResetVoc1               ; 
                        jsr ResetVoc3               ; 
                        
                        lda TabFilterCutHi          ; 
                        sta CUTHI                   ; SID($D416) Filter Cutoff Frequency (high byte)
                        
                        lda #$06                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        
                        lda #$0d                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
                        lda #SP_Ptr_Hill_45         ; enemy bullet phase 1 - min
                        sta BH_SpritePointer02_A    ; 
                        
                        ldx #$02                    ; 
                        jsr EnableSpriteNoFromXr    ; 
HillCannonAutoTargetX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonAutoFire    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonAutoFire      subroutine                  ; 
                        lda TabFlagTankEneShoot     ; 
                        bne .ChkWaitAutoAction      ; TabFlagTankEneShootYes
                        
.Exit                   rts                         ; 
                        
.ChkWaitAutoAction      lda TabWaitFlagAutoAction   ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagAutoAction   ; TabWaitFlag_Yes
                        
                        lda BH_SpritePointer02_A    ; 
                        cmp #SP_Ptr_Hill_4d         ; enemy bullet phase 9 - max
                        beq .HillTankHit            ; 
                        
                        inc BH_SpritePointer02_A    ; 
                        
                        rts                         ; 
                        
.HillTankHit            lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
.LightsOut              lda #BLACK                    ; 
                        sta BGCOL0                  ; VIC($D021) Background Color 0
                        
                        jsr ClearGfxScreen          ; 
                        
                        lda #$02                    ; 
                        sta TabWaitTimeTOD_Sec      ; 
                        jsr WaitTOD1_Sec            ; 
                        
                        lda TabBeachSpriteLevel     ; 
                        beq .NextTank               ; 
                        
                        dec TabBeachSpriteLevel     ; 
                        
.NextTank               pla                         ; 
                        pla                         ; 
                        jsr TankCheckLives          ; 
HillCannonAutoFireX     jmp TankInitAll             ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonPlayerFire  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonPlayerFire    subroutine                  ; 
                        lda #CIA_Joy_Fi             ; 
                        bit TabJoysticAction        ; 
                        bne .ClrJoyStickAction      ; 
                        
.Exit                   rts                         ; 
                        
.ClrJoyStickAction      jsr ClearDirInJoyAction     ; 
                        
                        lda TabP_ShotStatus         ; 
                        bne .Exit                   ; TabSpriteStatusUsed
                        
                        lda #SP_Ptr_Hill_55         ; my bullet phase 8 - max
                        sta BH_SpritePointer01_A    ; 
                        
                        lda #BH_SprtColorHillMyShot ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        
                        lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP0Y                    ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
                        sec                         ; 
                        sbc TabCannonShotDegree     ; 
                        sta TabBulletHight          ; 
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$fd                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        lda #$01                    ; 
                        and MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        asl a                       ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        inc TabP_ShotStatus         ; TabSpriteStatusUsed
                        
                        ldx #$01                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        jsr SetVoc2_Cannon          ; 
                        jsr PlayerAmmoDec           ; 
                        
                        lda #$01                    ; 
                        sta TabWaitTimeHillShotSav  ; 
                        sta TabWaitTimeHillShot     ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveEnemy    ; 
                        
                        lda #$00                    ; 
                        sta GR_SpriteStepCount      ; 
                        
HillCannonPlayerFireX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonPlayerShotFly Does  : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonPlayerShotFly subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkWaitMove            ; TabSpriteStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkWaitMove            lda TabWaitFlagMoveEnemy    ; 
                        bne .Exit                   ; 
                        
                        inc TabWaitFlagMoveEnemy    ; TabWaitFlag_Yes
                        inc GR_SpriteStepCount      ; 
                        
                        lda GR_SpriteStepCount      ; 
                        cmp #$2e                    ; 
                        bne .BulletSprtPtr          ; 
                        
                        ldx #$01                    ; .......#
                        jsr DisableSpriteNoFromXr   ; 
                        
                        dec TabP_ShotStatus         ; TabSpriteStatusFree
                        rts                         ; 
                        
.BulletSprtPtr          lda BH_SpritePointer01_A    ; 
                        cmp #SP_Ptr_Hill_4e         ; my bullet phase 1 - min
                        beq .BulletFly              ; 
                        
                        lda GR_SpriteStepCount      ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        sta TabWrkValueSav          ; 
                        
                        lda #SP_Ptr_Hill_55         ; my bullet phase 8 - max
                        sec                         ; 
                        sbc TabWrkValueSav          ; 
                        sta BH_SpritePointer01_A    ; 
                        
.BulletFly              lda TabBulletHight          ; 
                        lsr a                       ; 
                        lsr a                       ; 
                        sta TabWrkPlayerShot        ; 
                        
                        lda TabBulletHight          ; 
                        sec                         ; 
                        sbc TabWrkPlayerShot        ; 
                        sta TabBulletHight          ; 
                        
                        lda SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        sec                         ; 
                        sbc TabWrkPlayerShot        ; 
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
HillCannonPlayerShotFlyX rts                        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonCheckHits   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonCheckHits     subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkSprtSprtColl        ; TabSpriteStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkSprtSprtColl        lda #$02                    ; ......#.
                        bit TabSprtSprtCollReg      ; 
                        beq .Exit                   ; 
                        
                        lda #$10                    ; ...#....
                        bit TabSprtSprtCollReg      ; 
                        beq .Exit                   ; 
                        
                        lda GR_SpriteStepCount      ; 
                        cmp #$19                    ; 
                        bcc .Exit                   ; 
                        
                        ldx #$01                    ; .......#
                        jsr DisableSpriteNoFromXr   ; 
                        
                        dec TabP_ShotStatus         ; TabSpriteStatusFree
                        
.Score                  jsr AddScore_2000           ; 
                        
                        dec TabPlayerHillTargsToHit ; 
                        beq .GetAmount              ; victory
                        
                        jsr SetVoc3_Cannon          ; 
                        jsr HillCannonNextTarget    ; set next hill target
                        rts                         ; 
                        
.GetAmount              ldx TabSkillLevel           ; 
.ScoreLoop              jsr AddScore_9000           ; 
                        jsr AddScore_1000           ; 
                        
                        dex                         ; 
                        bpl .ScoreLoop              ; 
                        
                        jsr ScreenStatusShowScore   ; 
                        jsr HillCannonCapExplode    ; 
                        jsr HillCannonWaveFlag      ; 
                        
                        lda #$01                    ; 
HillCannonCheckHitsX    jmp TankCheckGameOver       ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonPlayerShotExp Does  : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonPlayerShotExp subroutine                  ; 
                        lda TabP_ShotStatus         ; 
                        bne .ChkFlyDist             ; TabSpriteStatusUsed
                        
.Exit                   rts                         ; 
                        
.ChkFlyDist             lda GR_SpriteStepCount      ; 
                        cmp #$1e                    ; 
                        bcc .Exit                   ; 
                        
                        lda #SP_Ptr_Hill_28         ; explosion hill target hit
                        sta BH_SpritePointer07_A    ; 
                        
                        lda #BH_SprtColorHillHitExplode ; 
                        sta SP7COL                  ; VIC($D02E) Color Sprite 7
                        
                        lda SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        sta SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        
                        lda MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        and #$7f                    ; 
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
                        and #$01                    ; 
                        beq .SfxExplode             ; 
                        
                        lda #$80                    ; 
                        ora MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        
.SfxExplode             jsr SetVoc3_Cannon          ; 
                        
                        lda #$01                    ; 
                        sta TabTargetHit            ; TabTargetHitYes
                        sta TabWaitFlagTargetHit    ; 
                        
                        ldx #$07                    ; .....###
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #$10                    ; 
                        sta TabWaitHillExplode      ; 
                        
                        dec TabP_ShotStatus         ; TabSpriteStatusFree
                        
                        ldx #$01                    ; .......#
                        jsr DisableSpriteNoFromXr   ; 
HillCannonPlayerShotExpX rts                        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonCapExplode  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonCapExplode    subroutine                  ; 
                        lda #$00                    ; 
                        sta XXPAND                  ; VIC($D01D) Sprite Double Height
                        sta YXPAND                  ; VIC($D017) Sprite Scale Double Width
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        ldx #$02                    ; 
.CreSpritePtrs          txa                         ; 
                        sec                         ; 
                        sbc #$02                    ; 
                        asl a                       ; 
                        asl a                       ; 
                        clc                         ; 
                        adc #SP_Ptr_Hill_2a         ; explosion hill cap piece  1
                        sta BH_SpritePointers_A,x   ; 
                        inx                         ; 
                        cpx #$07                    ; 
                        bne .CreSpritePtrs          ; 
                        
                        lda #SP_Ptr_Hill_29         ; explosion hill cap 1
                        sta BH_SpritePointer07_A    ; 
                        
                        lda #$42                    ; 
                        sta SP2Y                    ; VIC($D005) Sprite 2 Y-Pos (Bits 0-7)
                        sta SP3Y                    ; VIC($D007) Sprite 3 Y-Pos (Bits 0-7)
                        sta SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        sta SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        
                        lda #$4a                    ; 
                        sta SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        sta SP6Y                    ; VIC($D00D) Sprite 6 Y-Pos (Bits 0-7)
                        
                        lda #$a8                    ; 
                        sta SP2X                    ; VIC($D004) Sprite 2 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP5X                    ; VIC($D00A) Sprite 5 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$b0                    ; 
                        sta SP3X                    ; VIC($D006) Sprite 3 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP7X                    ; VIC($D00E) Sprite 7 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$b8                    ; 
                        sta SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        sta SP6X                    ; VIC($D00C) Sprite 6 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        ldx #$02                    ; 
                        lda #BH_SprtColorHillCapPieces ; 
.SetSpriteColors        sta SP0COL,x                ; VIC($D027) Color Sprite 0
                        inx                         ; 
                        cpx #$07                    ; 
                        bne .SetSpriteColors        ; 
                        
                        lda #$01                    ; 
                        sta SPMC0                   ; VIC($D025) Sprite Multicolor Register 0
                        
                        lda #$00                    ; 
                        sta SPMC1                   ; VIC($D026) Sprite Multicolor Register 1
                        
                        lda #BH_SprtColorHillCapExplode ; 
                        sta SP7COL                  ; VIC($D02E) Color Sprite 7
                        
                        lda #$7c                    ; 
                        sta SPMC                    ; VIC($D01C) Sprite Multicolor
                        
                        lda #$30                    ; 
                        sta TabMoveTorpedo_06       ; 
                        
                        lda #$28                    ; 
                        sta TabMoveTorpedo_04       ; 
                        sta TabMoveTorpedo_Val_1    ; 
                        
                        lda #$14                    ; 
                        sta TabMoveTorpedo_07       ; 
                        
                        ora #$80                    ; 
                        sta TabMoveTorpedo_03       ; 
                        
                        lda #$3b                    ; 
                        sta TabMoveTorpedo_09       ; 
                        
                        ora #$80                    ; 
                        sta TabMoveTorpedo_08       ; 
                        
                        lda #$00                    ; 
                        sta TabMoveTorpedo_Val_2    ; 
                        sta TabMoveTorpedo_Val_3    ; 
                        
                        lda #$45                    ; 
                        sta SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        
                        lda #$04                    ; 
                        sta TabWaitTimeCapExplSav   ; 
                        sta TabWaitTimeCapExpl      ; 
                        
                        lda #$3c                    ; 
                        sta TabWaitHillExplode      ; 
                        
                        lda #$02                    ; 
                        sta TabWaitTimeExplCapSav   ; 
                        sta TabWaitTimeExplCap      ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagExplodeCap   ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveEnemy    ; 
                        sta TabWaitFlagTargetHit    ; 
                        
                        lda #$fd                    ; ######.#
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        lda #$5c                    ; 
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapTop + $00 ; $488b
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapTop + $01 ; $488c
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapTop + $02 ; $488d
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapTop + $03 ; $488e
                        
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapBot + $00 ; $48b2
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapBot + $01 ; $48b3
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapBot + $02 ; $48b4
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapBot + $03 ; $48b5
                        sta BH_Scrn_Gfx + BH_Scrn_Gfx_OffHillCapBot + $04 ; $48b6
                        
                        lda #[COLORAM_MCM_On | CYAN]; color sky behind cap ruin
                        sta COLORAM + BH_Scrn_Gfx_OffHillCapBot + $01 ; $d8b3
                        sta COLORAM + BH_Scrn_Gfx_OffHillCapBot + $02 ; $d8b4
                        sta COLORAM + BH_Scrn_Gfx_OffHillCapBot + $03 ; $d8b5
                        
                        jsr ResetVoc1               ; 
                        jsr ResetVoc2               ; 
                        jsr ResetVoc3               ; 
                        
                        lda TabFilterCutHi          ; 
                        sta CUTHI                   ; SID($D416) Filter Cutoff Frequency (high byte)
                        
                        lda #$08                    ; 
                        sta FREHI1                  ; SID($D401) Oscillator 1 Freq Ctrl (high byte)
                        lda #$07                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        lda #$09                    ; 
                        sta FREHI3                  ; SID($D40F) Oscillator 3 Freq Ctrl (high byte)
                        
                        lda #$0e                    ; 
                        sta ATDCY1                  ; SID($D405) Oscillator 1 Attack/Decay2
                        sta ATDCY2                  ; SID($D40C) Oscillator 2 Attack/Decay
                        sta ATDCY3                  ; SID($D413) Oscillator 3 Attack/Decay
                        
                        lda #$81                    ; 
                        sta VCREG1                  ; SID($D404) Oscillator 1 Control
                        sta VCREG2                  ; SID($D40B) Oscillator 2 Control
                        sta VCREG3                  ; SID($D412) Oscillator 3 Control
                        
                        lda #$7c                    ; .#####..
.Init                   jsr AcToTabSpritesActive    ; 
                        
.HillCapExplode_Loop    jsr HillCapPiecesMoveUpDo   ; 
                        jsr HillCapSetSprtPtrs      ; 
                        jsr HillCapPiecesChkMaxY    ; -> with a return 
                        jsr HillCapSetBlack         ; 
                        
HillCannonCapExplodeX   jmp .HillCapExplode_Loop    ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCapPiecesMoveUpDo Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCapPiecesMoveUpDo   subroutine                  ; 
                        lda TabWaitFlagExplodeCap   ; 
                        beq .SetCapExplodeWaitFlag  ; TabWaitFlag_No
                        
.Exit                   rts                         ; 
                        
.SetCapExplodeWaitFlag  inc TabWaitFlagExplodeCap   ; TabWaitFlag_Yes
                        
                        ldx #$02                    ; 
.GetNextMoveUpDo        lda TabMoveSpriteUpDo,x     ; 
                        bmi .ChkMoveDoMax           ; TabMoveSprite_Do
                        beq .SetMoveDo              ; TabMoveSpriteUpDo_None = no value
                        
                        dec TabMoveSpriteUpDo,x     ; TabMoveSprite_Up
                        
.SetNextMoveUpDo        inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextMoveUpDo        ; 
                        
.Exit_1                 rts                         ; 
                        
.SetMoveDo              ora #TabMoveSprite_Do       ; 
                        sta TabMoveSpriteUpDo,x     ; 
                        jmp .SetNextMoveUpDo        ; 
                        
.ChkMoveDoMax           cmp #$bb                    ; 
                        beq .SetNextMoveUpDo        ; 
                        
                        inc TabMoveSpriteUpDo,x     ; 
HillCapPiecesMoveUpDoX  jmp .SetNextMoveUpDo        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCapSetSprtPtrs    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCapSetSprtPtrs      subroutine                  ; 
                        lda TabWaitFlagMoveEnemy    ; 
                        beq .SetMoveWaitFlag        ; 
                        
                        rts                         ; 
                        
.SetMoveWaitFlag        inc TabWaitFlagMoveEnemy    ; TabWaitFlag_Yes
                        
                        ldx #$02                    ; 
.GetNextPieceSprite     lda TabSpriteActiveTank,x   ; 
                        beq .SetNextPieceSprite     ; 
                        
                        lda BH_SpritePointer03_A    ; 
                        cmp #SP_Ptr_Hill_31         ; explosion hill cap piece  8
                        beq .SetNextPieceSpritePtr  ; 
                        
                        inc BH_SpritePointers_A,x   ; 
                        jmp .SetNextPieceSprite     ; 
                        
.SetNextPieceSpritePtr  txa                         ; 
                        sec                         ; 
                        sbc #$02                    ; 
                        asl a                       ; 
                        asl a                       ; 
                        clc                         ; 
                        adc #SP_Ptr_Hill_2a         ; explosion hill cap piece  1
                        sta BH_SpritePointers_A,x   ; 
                        
.SetNextPieceSprite     inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextPieceSprite     ; 
                        
HillCapSetSprtPtrsX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCapPiecesChkMaxY  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCapPiecesChkMaxY    subroutine                  ; 
                        lda #$00                    ; 
                        sta TabTankObstOffNext      ; 
                        
                        ldx #$02                    ; 
.GetNextPiesSpriteOff   jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0Y,y                  ; VIC($D001) Sprite 0 Y-Pos (Bits 0-7)
                        cmp TabPiecesMaxPosY,x      ; 
                        bcc .IncOff                 ; lower
                        
                        lda #$00                    ; 
                        sta TabSpriteActiveTank,x   ; 
                        jmp .SetNextPiesSpriteOff   ; 
                        
.IncOff                 inc TabTankObstOffNext      ; 
                        
.SetNextPiesSpriteOff   inx                         ; 
                        cpx #$07                    ; 
                        bne .GetNextPiesSpriteOff   ; 
                        
                        lda TabTankObstOffNext      ; 
                        bne HillCapPiecesChkMaxYX   ; 
                        
                        pla                         ; 
                        pla                         ; 
HillCapPiecesChkMaxYX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabPiecesMaxPosY        dc.b $00 ; unused
                        dc.b $00 ; unused
                        dc.b $98 ; 
                        dc.b $a2 ; 
                        dc.b $9c ; 
                        dc.b $88 ; 
                        dc.b $8c ; 
                        dc.b $60 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCapSetBlack       Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCapSetBlack         subroutine                  ; 
                        lda TabWaitFlagTargetHit    ; 
                        beq .ColorCap               ; TabWaitFlag_No
                        
                        rts                         ; 
                        
.ColorCap               lda #BH_SprtColorHillCapRuin; 
                        sta SP7COL                  ; VIC($D02E) Color Sprite 7
                        
HillCapSetBlackX        rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonWaveFlag    Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonWaveFlag      subroutine                  ; 
                        ldx #$02                    ; 
.ChkWaveTime            lda TabMoveSpriteLeRiTime,x ; 
                        beq .SetNextFlagPos         ; 
                        
                        rts                         ; 
                        
.SetNextFlagPos         inx                         ; 
                        cpx #$07                    ; no flags + $01
                        bne .ChkWaveTime            ; 
                        
                        lda #SP_Ptr_Hill_3e         ; wave white flag right step 1 - max left
                        sta BH_SpritePointer01_A    ; 
                        
                        lda #$0a                    ; 
                        sta TabWaitTimeHillFlagSav  ; 
                        sta TabWaitTimeHillFlag     ; 
                        
                        lda #BH_SprtColorHillResignFlag ; 
                        sta SP1COL                  ; VIC($D028) Color Sprite 1
                        
                        lda #$44                    ; 
                        sta SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        
                        lda #$b0                    ; 
                        sta SP1X                    ; VIC($D002) Sprite 1 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        ldx #$01                    ; 
                        jsr EnableSpriteNoFromXr    ; 
                        
                        lda #TabWaitFlag_Yes        ; 
                        sta TabWaitFlagMoveEnemy    ; 
                        jsr ResetTOD1               ; 
                        
.Wave                   lda TabWaitFlagMoveEnemy    ; 
                        bne .Wave                   ; 
                        
                        inc TabWaitFlagMoveEnemy    ; TabWaitFlag_Yes
                        
                        lda BH_SpritePointer01_A    ; 
                        cmp #SP_Ptr_Hill_43         ; wave white flag left step 3
                        beq .GetMaxWaveLeft         ; 
                        
                        inc BH_SpritePointer01_A    ; 
                        jmp .ChkTimer               ; 
                        
.GetMaxWaveLeft         lda #SP_Ptr_Hill_3e         ; wave white flag right step 1 - max left
                        sta BH_SpritePointer01_A    ; 
                        
.ChkTimer               lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        cmp #$10                    ; 
                        bcc .Wave                   ; 
                        
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
HillCannonWaveFlagX     rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillDemo              Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillDemo                subroutine                  ; 
                        lda TabGameMode             ; 
                        bne .ChkDemoTime            ; TabGameModeDemo or TabGameModeExit
                        
                        rts                         ; 
                        
.ChkDemoTime            lda TO2SEC                  ; CIA2($DD09) Time of Day Clock Seconds
                        cmp #$10                    ; 
                        bne .ChkCannonPosX_Msb      ; 
                        
                        jsr ResetTOD2               ; 
                        
                        pla                         ; level1
                        pla                         ; 
                        
                        pla                         ; level2
                        pla                         ; 
                        
                        lda #TabShowScoresYes       ; 
                        sta TabPlayerShowScores     ; 
                        
                        rts                         ; 
                        
.ChkCannonPosX_Msb      lda #$01                    ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .ChkTargPosX            ; 
                        
.SetCannonMov_Le        lda #CIA_Joy_Le             ; 
                        sta TabJoysticAction        ; 
                        
                        jmp .ChkTargPosY            ; 
                        
.ChkTargPosX            lda SP0X                    ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp SP4X                    ; VIC($D008) Sprite 4 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        beq .ChkTargPosY            ; 
                        bcs .SetCannonMov_Le        ; 
                        
                        lda #CIA_Joy_Ri             ; 
                        sta TabJoysticAction        ; 
                        
.ChkTargPosY            lda TabCannonShotDegree     ; 
                        clc                         ; 
                        adc #$04                    ; 
                        cmp SP4Y                    ; VIC($D009) Sprite 4 Y-Pos (Bits 0-7)
                        beq .SetCannonMov_Fi        ; target fully aimed
                        bcs .GetCannonMov_Do        ; 
                        
.GetCannonMov_Up        lda #CIA_Joy_Up             ; 
                        
.SetCannonMov_UpDo      ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
.SetCannonMov_Fi        lda #CIA_Joy_Fi             ; 
                        ora TabJoysticAction        ; 
                        sta TabJoysticAction        ; 
                        
                        rts                         ; 
                        
.GetCannonMov_Do        lda #CIA_Joy_Do             ; 
HillDemoX               jmp .SetCannonMov_UpDo      ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveSetLeft Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveSetLeft   subroutine                  ; 
                        lda TabMoveSpriteLeRiTime   ; 
                        bmi MapMyFleetMoveSetLeftX  ; 
                        
                        dec TabMoveSpriteLeRiTime   ; 
                        
MapMyFleetMoveSetLeftX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveSetRight Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveSetRight  subroutine                  ; 
                        lda TabMoveSpriteLeRiTime   ; 
                        cmp #$01                    ; 
                        beq MapMyFleetMoveSetRightX ; 
                        
                        inc TabMoveSpriteLeRiTime   ; 
                        
MapMyFleetMoveSetRightX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveSetUp   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveSetUp     subroutine                  ; 
                        lda TabMoveSpriteUpDoTime   ; 
                        cmp #$01                    ; 
                        beq MapMyFleetMoveSetUpX    ; 
                        
                        inc TabMoveSpriteUpDoTime   ; 
                        
MapMyFleetMoveSetUpX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; MapMyFleetMoveSetDown Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
MapMyFleetMoveSetDown   subroutine                  ; 
                        lda TabMoveSpriteUpDoTime   ; 
                        bmi MapMyFleetMoveSetDownX  ; 
                        
                        dec TabMoveSpriteUpDoTime   ; 
                        
MapMyFleetMoveSetDownX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; AcToTabSpritesActive  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
AcToTabSpritesActive    subroutine                  ; 
                        ldy #$07                    ; bit count
.ShiftNextBit           asl a                       ; shift a bit to carry
                        pha                         ; save ac
                        
                        lda #TabSpriteActiveNo      ; preset no
                        bcc .StoreBitInTab          ; 
                        
                        lda #TabSpriteActiveYes     ; 
                        
.StoreBitInTab          sta TabSpriteActiveAll,y    ; 
                        
                        pla                         ; restore ac
                        dey                         ; bit count
                        bpl .ShiftNextBit           ; 
                        
AcToTabSpritesActiveX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ClearGfxRow_20        Does    : Shootout Graphic Screen one row smaller
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ClearGfxRow_20          subroutine                  ; 
                        ldx #$27                    ; 
.Clear                  lda #$4f                    ; 
                        sta BH_Scrn_Gfx + BH_Scrn_Row_14,x ; 
                        
                        lda #BLACK                  ; 
                        sta COLORAM + $02f8,x       ; 
                        
                        dex                         ; 
                        bpl .Clear                  ; 
                        
ClearGfxRow_20X         rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; ScreenStatusDestroy   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
ScreenStatusDestroy     subroutine                  ; 
                        lda TabSeaShipMsgDestroy    ; 
                        cmp #TabSeaShipMsgDestroyYes; 
                        beq .MsgOut                 ; 
                        
                        rts                         ; 
                        
.MsgOut                 lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        cmp #$05                    ; 
                        beq .MsgClear               ; 
                        
                        lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        sta TabBackGroundColor      ; 
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapDestr ; 
                        ldx #<TabTextStatusShipDestr; 
                        jsr ScreenStatusTextOutRow3 ; xr=input text offset  yr=output text offset
                        rts                         ; 
                        
.MsgClear               lda #$00                    ; 
                        sta TODHRS                  ; CIA1($DC0B) Time of Day Clock Hours
                        sta TabSeaShipMsgDestroy    ; TabSeaShipMsgDestroyNo
                        
                        lda #CYAN                   ; 
                        sta TabBackGroundColor      ; 
                        
                        lda #$ff                    ; flag: blank
                        sta TabScrnStatusTextColor  ; 
                        
                        ldy #BH_Scrn_Stat_OffMapDestr ; 
                        ldx #BH_Scrn_Stat_LenMapDestr ; 
                        jsr ScreenStatusTextOutRow3 ; xr=length blank out  yr=output text offset
ScreenStatusDestroyX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipStatusShipsOut Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipStatusShipsOut   subroutine                  ; 
                        jsr ScreenStatusShowShips   ; 
                        
                        lda CHR_LOR                 ; Char ROM($dc00) lower case reversed
                        eor #$ff                    ; 
                        and #$03                    ; ......##
                        bne .ChkSpriteStatus        ; 
                        
                        lda #TabSpriteStatusFree    ; 
                        sta TabSpriteStatus         ; 
                        
                        lda #$14                    ; 
                        sta TabCountDown1_00        ; 
                        
                        rts                         ; 
                        
.ChkSpriteStatus        lda TabSpriteStatus         ; 
                        bne .ChkWait                ; TabSpriteStatusUsed
                        
                        lda #$04                    ; 
                        sta TabPlaySfxVoc1_06       ; 
                        sta TabPlaySfxVoc2_06       ; 
                        
                        lda #$01                    ; 
                        sta TabSpriteStatus         ; TabSpriteStatusUsed
                        sta TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
.Exit                   rts                         ; 
                        
.ChkWait                lda TabWaitFlagSfxShot      ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSfxShot      ; TabWaitFlag_Yes
                        
                        lda TabCountDown1_00        ; 
                        cmp #$01                    ; 
                        beq .Exit                   ; 
                        
                        dec TabCountDown1_00        ; 
                        
SeaShipStatusShipsOutX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEnemyGetCollSpBg Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEnemyGetCollSpBg subroutine                  ; 
                        lda SP1Y                    ; VIC($D003) Sprite 1 Y-Pos (Bits 0-7)
                        cmp #$98                    ; 
                        bcc .GoGetColl              ; lower
                        
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        ora TabSprtBackCollReg      ; 
                        sta TabSprtBackCollReg      ; 
                        
                        rts                         ; 
                        
.GoGetColl              jsr GetSprtBackCollReg      ; sprite/foregr collision
SeaShipEnemyGetCollSpBgX rts                        ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipChkDestroyMsg  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipChkDestroyMsg    subroutine                  ; 
                        lda TabSeaShipMsgDestroy    ; 
                        cmp #TabSeaShipMsgDestroyYes; 
                        beq .MsgDestroyOut          ; 
                        
                        rts                         ; 
                        
.MsgDestroyOut          jsr ScreenStatusDestroy     ; 
                        
                        lda TabSeaShipMsgDestroy    ; 
                        
SeaShipChkDestroyMsgX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipSinkAndScore   Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipSinkAndScore     subroutine                  ; 
                        lda TabTargetShipHit        ; 
                        bpl .ChkCarrier             ; TabTargetShipHitCarr
                        
                        rts                         ; 
                        
.ChkCarrier             cmp #$05                    ; 
                        beq .ChkSinks               ; 
                        
                        tax                         ; 
                        jsr SeaShipVesselScore      ; 
                        rts                         ; 
                        
.ChkSinks               lda TabWaitFlagSinkShip     ; 
                        bne SeaShipSinkAndScoreX    ; TabWaitFlag_Yes
                        
                        inc TabWaitFlagSinkShip     ; TabWaitFlag_Yes
                        
                        inc SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        inc SP6Y                    ; VIC($D00D) Sprite 6 Y-Pos (Bits 0-7)
                        inc SP7Y                    ; VIC($D00F) Sprite 7 Y-Pos (Bits 0-7)
                        
                        lda SP5Y                    ; VIC($D00B) Sprite 5 Y-Pos (Bits 0-7)
                        cmp #$9b                    ; 
                        bne SeaShipSinkAndScoreX    ; 
                        
SeaShipScoreCarrier     subroutine                  ; 
                        lda #$07                    ; .....###
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        jsr AddScore_1000           ; 
                        jsr AddScore_9000           ; 
                        
                        lda #$df                    ; ##.#####
                        and TabPlayerVesselActive   ; one bit per ship
                        sta TabPlayerVesselActive   ; 
                        
                        lda #TabTargetShipHitNo     ; 
                        sta TabTargetShipHit        ; 
                        
SeaShipSinkAndScoreX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SeaShipEneExplodeOff  Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SeaShipEneExplodeOff    subroutine                  ; 
                        lda TabTargetHit            ; 
                        bne .ChkTime                ; TabTargetHitYes
                        
                        rts                         ; 
                        
.ChkTime                lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        beq SeaShipEneExplodeOffX   ; 
                        
.ExplodeFlashOff        lda #CYAN                   ; sky
                        sta TabBackGroundColor      ; 
                        
                        lda TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp #$03                    ; 
                        bne SeaShipEneExplodeOffX   ; 
                        
.SpriteExplosionOff     lda #$fd                    ; ######.#
                        and SPENA                   ; VIC($D015) Sprite Enable Register
.Enab                   sta SPENA                   ; VIC($D015) Sprite Enable Register
                        
                        dec TabTargetHit            ; TabTargetHitNo
                        
SeaShipEneExplodeOffX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; NextEnemyBulletPhase  Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
NextEnemyBulletPhase    subroutine                  ; 
                        ldy #$00                    ; 
                        ldx TabEnemyBulletPhase     ; 
.GetNextBulletPase      inx                         ; 
                        cpx #$05                    ; max
                        bne .SetNextBulletPase      ; 
                        
                        cpy #$01                    ; 
                        bne .Set2ndRound            ; 
                        
.Clr                    lda #$ff                    ; 
                        sta TabEnemyBulletPhase     ; 
                        
                        rts                         ; 
                        
.Set2ndRound            iny                         ; 
                        
                        ldx #$01                    ; 
                        
.SetNextBulletPase      stx TabEnemyBulletPhase     ; 
                        
                        lda TabSeaVesselActive,x    ; 
                        beq .GetNextBulletPase      ; TabSeaVesselActiveNo
                        
NextEnemyBulletPhaseX   rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankAdaptSpritePosX   Does : Keep PosX despite beach scrolling
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankAdaptSpritePosX     subroutine                  ; 
                        ldx #$07                    ; 
.GetNextSprite          jsr SetYrToXrMult2          ; yr = xr * 2
                        lda TabTankAdaptPosX,x      ; 
                        bmi .Negative               ; 
                        
.Positive               lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc TabTankAdaptPosX,x      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$00                    ; 
                        sta TabTankAdaptPosX,x      ; 
                        bcc .SetNextSprite          ; 
                        
.SetNewPosX_Msb         jsr SetAcToXrTimesMult2     ; 
                        eor MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        sta MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        jmp .SetNextSprite          ; 
                        
.Negative               lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        clc                         ; 
                        adc TabTankAdaptPosX,x      ; 
                        sta SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        
                        lda #$00                    ; 
                        sta TabTankAdaptPosX,x      ; 
                        bcc .SetNewPosX_Msb         ; 
                        
.SetNextSprite          dex                         ; 
                        bne .GetNextSprite          ; 
                        
TankAdaptSpritePosXX    rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankClearSpriteAtMaxX Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankClearSpriteAtMaxX   subroutine                  ; 
                        ldx #$06                    ; 
.ChkSpritePosX          jsr SetAcToXrTimesMult2     ; 
                        bit MSIGX                   ; VIC($D010) MSB X-Pos Sprites 0-7
                        beq .MaxPosX                ; 
                        
                        jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
.MaxPosX_Msb            cmp #$58                    ; 
                        bcc .SetNextSpriteNo        ; 
                        
.ClrSprite              lda #TabSpriteStatusFree    ; 
                        sta TabSpriteStatus,x       ; 
                        jsr SetAcToXrTimesMult2     ; 
                        
                        eor #$ff                    ; 
                        and SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        jmp .SetNextSpriteNo        ; 
                        
.MaxPosX                jsr SetYrToXrMult2          ; yr = xr * 2
                        lda SP0X,y                  ; VIC($D000) Sprite 0 X-Pos (Bits 0-7 - Bit 8 in MSIGX = $D010)
                        cmp #$e0                    ; 
                        bcc .SetNextSpriteNo        ; 
                        
                        cpx TabTankShotSprtNoSav    ; 
                        bne .SetNextSpriteNo        ; 
                        
                        lda #$00                    ; 
                        sta TabTankShotSprtNoSav    ; 
                        
.SetNextSpriteNo        dex                         ; 
                        cpx #$02                    ; 
                        bne .ChkSpritePosX          ; 
                        
TankClearSpriteAtMaxXX  rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; TankInitSfx           Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
TankInitSfx             subroutine                  ; 
                        jsr ResetTOD1               ; 
.GoAdaptSfx             jsr TankAdaptSfxWithPosY    ; 
                        
                        lda #$01                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        
                        lda TODSEC                  ; CIA1($DC09) Time of Day Clock Seconds
                        ldx TODTEN                  ; CIA1($DC08) Time of Day Clock Tenths
                        cmp TabWaitTimeTOD_Sec      ; 
                        bne .GoAdaptSfx             ; 
                        
                        lda #$02                    ; 
                        sta FREHI2                  ; SID($D408) Oscillator 2 Freq Ctrl (high byte)
                        
                        lda #$00                    ; 
                        sta TabJoysticAction        ; 
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        
TankInitSfxX            rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; HillCannonPlayerExpClr Does   : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
HillCannonPlayerExpClr  subroutine                  ; 
                        lda TabTargetHit            ; 
                        bne .ChkWaitHit             ; TabTargetHitYes
                        
.Exit                   rts                         ; 
                        
.ChkWaitHit             lda TabWaitFlagTargetHit    ; 
                        bne .Exit                   ; TabWaitFlag_Yes
                        
                        dec TabTargetHit            ; TabTargetHitNo
                        
                        ldx #$07                    ; .....###
                        jsr DisableSpriteNoFromXr   ; 
HillCannonPlayerExpClrX rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
; SpritesReset          Does    : 
;                       Expects : 
;                       Returns : 
; -------------------------------------------------------------------------------------------------------------- ;
SpritesReset            subroutine                  ; 
                        lda #$00                    ; 
                        sta SPENA                   ; VIC($D015) Sprite Enable Register
                        sta SPBGPR                  ; VIC($D01B) Sprite to Foreground Display Priority
                        sta TabJoysticAction        ; 
                        
                        lda SPSPCL                  ; VIC($D01E) Sprite-Sprite Collision - Cleared on read
                        lda SPBGCL                  ; VIC($D01F) Sprite-Foreground Collision - Cleared on read
                        
SpritesResetX           rts                         ; 
; -------------------------------------------------------------------------------------------------------------- ;
                        * equ $9600                 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabSkillLevel           dc.b $01 ; .hbu001. - BASIC 120
TabNumberOfPlayers      dc.b $01 ; .hbu001. - BASIC 120
TabP_ShowScores         dc.b $01 ; player1
                        dc.b $00 ; player2
TabActualPlayerNumber   dc.b $00 ; 
TabScrnStatusTextColor  dc.b $00 ; 
TabPlayerScoreLo        dc.b $20 ; 
TabPlayerScoreHi        dc.b $01 ; 
TabPlayerShowScores     dc.b $02 ; .hbu001. - BASIC 216
TabShowScoresNo            = $00 ; 
TabShowScoresYes           = $02 ; 
TabPlayerCavePassed     dc.b $81 ; 
TabCavePassedIni           = $00 ; 
TabCavePassedYesGame       = $01 ; entered bay via cave
TabCavePassedNo            = $80 ; entered bay directly
TabCavePassedYesChase      = $81 ; entered bay via cave
TabMyFleetPosX          dc.b $85 ; 
TabMyFleetPosY          dc.b $ad ; 
TabMyFleetMsbPosX       dc.b $00 ; 
TabPlanes2Shoot         dc.b $0a ; 
TabPlayerLives          dc.b $02 ; 
TabPlayerTanksLeft      dc.b $00 ; 
TabPlayerHillTargsToHit dc.b $0a ; 
TabBayMapSeaChapters    dc.b $01 ; 
TabBayMapSeaChaptersNo    = $00  ; storm the hill chapters
TabBayMapSeaChaptersYes   = $01  ; bay shoot outs
TabPlayerShips          dc.b $02 ; 
TabCaveShipSpeed        dc.b $07 ; 
TabCaveShipSpeedMin       = $00  ; 
TabCaveShipSpeedMax       = $07  ; 
TabCaveShipSpeedInit      = TabCaveShipSpeedMin ; 
TabCaveShipExits        dc.b $00 ; 
TabCaveShipExitsNo        = $00  ; 
TabCaveShipExitsYes       = $01  ; 
TabSprtSprtCollReg      dc.b $00 ; 
TabSprtBackCollReg      dc.b $00 ; 
TabWaitTimeTOD_Sec      dc.b $05 ; 
TabWaitTimeTOD_10th     dc.b $09 ; 
TabPlayerVesselActive   dc.b $3e ; one bit per ship

TabPlrShootSprtBasePtr2 dc.b $4d ; 

TabCannonShotDegree     dc.b $60 ; 
TabCannonShotHeight     dc.b $9c ; 
TabCannonShotDist       dc.b $60 ; 

TabPlayerShoots         dc.b $00 ; 
TabPlayerShootsNo         = $00  ; 
TabPlayerShootsYes        = $01  ; 

TabPlayerShootSpriteNo  dc.b $01 ; 

TabBulletHight          dc.b $00 ; 
                        dc.b $00 ; bullet1
                        dc.b $00 ; bullet2
                        dc.b $00 ; 

TabDemoPlaneMyPosX      dc.b $00 ; cannon
TabDemoPlaneMyPosX_MSB  dc.b $00 ; 
TabDemoPlanePosX        dc.b $00 ; plane
TabDemoPlanePosX_MSB    dc.b $00 ; 

TabPlrShootSprtBasePtr1 dc.b $41 ; 

TabIRQScreenPartFlag    dc.b $00 ; 
TabIRQScreenPartGfx        = $01 ; 
TabIRQScreenPartStatus     = $00 ; 
TabTankActiveScreen     dc.b $01 ; screen A
TabVicMemControl        dc.b $20 ; 
TabVicMemControlSetA    dc.b $20 ; ..#. ... o - Scrn_adr=$0800-$0be7($02) Chars=$0000-$07ff
TabVicMemControlSetB    dc.b $30 ; ..## ... o - Scrn_adr=$0c00-$0fe7($03) Chars=$0000-$07ff
TabBackGroundColor      dc.b YELLOW
TabBackGroundColorSav   dc.b $00 ; 
TabVicScrollSoftPosY    dc.b $03 ; 
TabTankScrollSoftPosY   dc.b $00 ; 
TabVicScrollSoftPosX    dc.b $00 ; 
TabTankScrollSoftPosX   dc.b $03 ; 
TabScreenTankColorFlag  dc.b $00 ; 
TabScreenTankColorNo       = $00 ; 
TabScreenTankColorYes      = $ff ; 
TabScrnTankPosY         dc.b $14 ; 
TabScrnTankPosYSky         = $00 ; 
TabScrnTankPosYTreeTop     = $05 ; 
TabScrnTankPosYTreeLeaf    = $06 ; 
TabScrnTankPosYTreeTwig    = $07 ; 
TabScrnTankPosYTreeTrunk   = $08 ; 
TabScrnTankPosYMax         = $14 ; 
TabTankObstPosY         dc.b $00 ; 
TabTankObstChrNum       dc.b $00 ; 
TabTankObstColor        dc.b $06 ; 
TabSaveCharDataByte     equ  *   ; 
TabSaveSpriteDataCopyXR equ  *   ; 
TabTankObstOffNext      dc.b $00 ; 
TabScrnTankPosX         dc.b $00 ; 
TabScrnTankPosXMax         = $26 ; 
TabScrnTankBayCharPtr   dc.b $04 ; 
TabScrnGfxActualMaxX       = BH_Scrn_RowLen ; 
TabScrnTankBayCharNext  dc.b $03 ; 
TabScrnTankBayCharNextIni  = $08 ; 
TabScrnTankBayScrollPos dc.b $ae ; 
TabScrnTankBayScrollPosIni = $d8 ; 
TabScrnTankBayScrollPosGo  = $ca ; 
TabScrnTankBayScrollDec dc.b $d5 ; 
TabTankScrollAreaTabPtr dc.b $00 ; 
TabTankScrollAreaTabPtrStart= $d8; 
TabTankScrollAreaTabPtrEnd  = $00; 
TabPlayerBulletPhaseNo  dc.b $00 ; 
TabScrnTankBayScrollFin dc.b $00 ; 
TabScrnTankBayScrollFinNo  = $ff ; 
TabScrnTankBayScrollFinYes = $00 ; 
TabCharDataPtrOutLo     dc.b $38 ; 
TabCharDataPtrOutHi     dc.b $04 ; 
TabTankSprtScrollSpeed  dc.b $01 ; 
                        dc.b $00 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
TabJoysticAction        dc.b $00 ; 
TabUserAction           dc.b $01 ; 
TabUserActionNo           = $00  ; 
TabUserActionYes          = $01  ; 

TabGameMode             dc.b $00 ; .hbu001. - BASIC 85/105/216
TabGameModeGame           = $00  ; 
TabGameModeDemoToCave     = $01  ; fleet moves to cave
TabGameModeDemoInBay      = $02  ; fleet encounters enemy fleet
TabGameModeDemoToBeach    = $03  ; fleet moves to beach
TabGameModeExit           = $ff  ; 

TabPlaneDirReLi         dc.b $00 ; 
TabPlaneDirReLiStop       = $00  ;
TabPlaneDirReLiCont       = $01  ;
TabPlaneDirReLi_2         = $02  ;
TabPlaneDirReLi_3         = $03  ;
TabPlaneDirReLi_4         = $04  ;
TabPlaneDirReLi_5         = $05  ;
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
TabPlaneVal_1           dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 

TabPlaneDirUpDo         dc.b $00 ; 
TabPlaneDirUpDo_0         = $00  ; 
TabPlaneDirUpDo_1         = $01  ; 
TabPlaneDirUpDo_2         = $02  ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
TabPlaneVal_2           dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 

TabPlaneFlies           dc.b $00 ; 
TabPlaneFliesNo           = $00  ; 
TabPlaneFliesYes          = $01  ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabMoveSpriteLeRiTime   dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabMoveSpriteUpDoTime   dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
T_9680                  dc.b $00 ; ???
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Wait Times: Start  of Wait Time Blocks
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlag_No            = $00  ; std global wait flag values
TabWaitFlag_Yes           = $01  ;
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block01    equ  *   ; IRQ_Exec_Waits: times block 01
TabWaitTimes            equ  *   ;                         
TabWaitTimePause        equ  *   ; 
TabWaitTimeExplTank     dc.b $01 ; 
TabWaitTimeHillFlag     equ  *   ; 
TabWaitTimeTankShot     equ  *   ; 
TabWaitTimeHillShot     equ  *   ; 
TabWaitTimeCapExpl      dc.b $01 ; 
TabWaitTimeHillShotE    equ  *   ; 
TabWaitWaterSplash      equ  *   ; 
TabWaitHillCannonAim    dc.b $01 ; 
TabWaitBulletFly        dc.b $01 ; 
TabWaitSinkShip         dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
TabWaitHillExplode      dc.b $01 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block02    equ  *   ; IRQ_Exec_Waits: times block 02
                        dc.b $02 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $06 ; 
                        dc.b $06 ; 
                        dc.b $06 ;
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block03    equ  *   ; IRQ_Exec_Waits: times block 03
TabWaitTimeMoveUpDo     dc.b $01 ; 
TabWaitTimeBullFlyPh    dc.b $01 ; 
TabWaitTimeBells        dc.b $01 ; 
TabWaitFlagBullFlyE     dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block04    equ  *   ; IRQ_Exec_Waits: times block 04
TabWaitTimePlaneFly     equ  *   ; 
TabWaitTimeExplCap      dc.b $02 ; 
                        dc.b $00 ; 
                        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $1e ; 
                        dc.b $1e ; 
                        dc.b $1e ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block05    equ  *   ; IRQ_Exec_Waits: times block 05
TabPlaySfxVoc1_00       dc.b $01 ; 
TabPlaySfxVoc1_01       dc.b $01 ; 
                        dc.b $01 ; 
TabPlaySfxVoc1_03       dc.b $01 ; 
TabPlaySfxVoc1_04       dc.b $01 ; 
TabPlaySfxVoc1_05       dc.b $01 ; 
TabPlaySfxVoc1_06       dc.b $01 ; 
                        dc.b $20 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Wait Times: Flags
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags            equ  *   ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags_Block01    equ  *   ; IRQ_Exec_Waits: flags block 01
TabWaitFlagPause        equ  *   ; 
TabWaitFlagExplTank     equ  *   ; 
TabWaitFlagMoveLeRi     dc.b $00 ; $01
TabWaitFlagMoveEnemy    dc.b $00 ; $02
TabWaitFlagAutoAction   dc.b $00 ; $03
TabWaitFlagBulletFly    dc.b $00 ; $04
TabWaitFlagSinkShip     dc.b $00 ; $05
TabWaitFlagMoveShip     dc.b $00 ; $06
                        dc.b $00 ; 
TabWaitFlagTargetHit    dc.b $00 ; $08
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags_Block02    equ  *   ; IRQ_Exec_Waits: flags block 02
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags_Block03    equ  *   ; IRQ_Exec_Waits: flags block 03
TabWaitFlagMoveUpDo     dc.b $00 ; $01
TabWaitFlagBulletFlyPh  dc.b $00 ; $02
TabWaitFlagPing         dc.b $00 ; $03
TabWaitFlagPlaneFly     dc.b $00 ; $04
                        dc.b $00 ; $05
                        dc.b $00 ; $06
                        dc.b $00 ; $07
                        dc.b $00 ; $08
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags_Block04    equ  *   ; IRQ_Exec_Waits: flags block 04
TabWaitFlagExplodeCap   dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitFlags_Block05    equ  *   ; IRQ_Exec_Waits: flags block 05
TabWaitFlagCaveShipLeRi dc.b $00 ; $01
TabWaitFlagCaveShipUpDo dc.b $00 ; $02
                        dc.b $00 ; $03
TabWaitFlagExplode      dc.b $00 ; $04
TabWaitFlagSfxShotPlane dc.b $00 ; $05
TabWaitFlagSfx          dc.b $00 ; $06
TabWaitFlagSfxShot      dc.b $00 ; $07
TabWaitFlagAmmoInc      dc.b $00 ; $08
; -------------------------------------------------------------------------------------------------------------- ;
; Wait Times Save Values for Restore: default=$01
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimesSav         equ  *   ; saved values for restore after waitime has run out
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block01Sav equ  *   ; IRQ_Exec_Waits: saves block 01
TabWaitTimeExplTankSav  equ  *   ; 
TabWaitTimeBayLeRiSav   equ  *   ; 
TabWaitTimeCanLeRiSav   equ  *   ; 
TabWaitTimeBeachLeRiSav dc.b $01 ; 
TabWaitTimeBayLeRiSavE  equ  *   ; 
TabWaitTimeHillFlagSav  equ  *   ; 
TabWaitTimeTankShotSav  equ  *   ; 
TabWaitTimeHillShotSav  equ  *   ; 
TabWaitTimeCapExplSav   dc.b $01 ; 
TabWaitTimeHillShotSavE equ  *   ; 
TabWaitTimeBayBlinkSav  equ  *   ; 
TabWaitHillCannonAimSav dc.b $01 ; 
TabWaitBulletFlySav     dc.b $01 ; 
TabWaitSinkShipSav      dc.b $01 ; 
TabWaitMoveCarrierSav   dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block02Sav equ  *   ; IRQ_Exec_Waits: saves block 02
TabWaitTimePlaneFlyLe   dc.b $02 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $06 ; 
                        dc.b $06 ; 
                        dc.b $06 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block03Sav equ  *   ; IRQ_Exec_Waits: saves block 03
TabWaitTimeBayUpDoSav   equ  *   ; 
TabWaitTimeCanUpDoSav   equ  *   ; 
TabWaitTimeBeachUpDoSav equ  *   ; 
TabCountDown1_00        dc.b $01 ; 
TabWaitTimeBayUpDoSavE  equ  *   ; 
TabWaitTimeBullFlyPhSav dc.b $01 ; 
TabWaitTimeBellsSav     dc.b $01 ; 
TabWaitFlagBullFlyESav  dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block04Sav equ  *   ; IRQ_Exec_Waits: saves block 04
TabWaitTimePlaneFlySav  equ  *   ; 
TabWaitTimeExplCapSav   dc.b $02 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $1e ; 
                        dc.b $1e ; 
                        dc.b $1e ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabWaitTimes_Block05Sav equ  *   ; IRQ_Exec_Waits: saves block 05
TabPlaySfxVoc2          equ  *   ; 
TabPlaySfxVoc2_00       dc.b $01 ; 
TabPlaySfxVoc2_01       dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
TabPlaySfxVoc2_05       dc.b $01 ; 
TabPlaySfxVoc2_06       dc.b $01 ; 
TabPlaySfxVoc2_07       dc.b $20 ; 
; -------------------------------------------------------------------------------------------------------------- ;
; Wait Times: End of Wait Time Blocks
; -------------------------------------------------------------------------------------------------------------- ;
TabMoveSprite_Le          = $80  ; bit7:flag: move left
TabMoveSprite_Ri          = $00  ; bit7:flag: move right
TabMoveSprite_Up          = $00  ; bit7:flag: move right
TabMoveSprite_Do          = $80  ; bit7:flag: move left

TabMoveSpriteLeRi       dc.b $00 ; cave and planes
TabMoveTorpedo_01       dc.b $00 ; 
TabMoveTorpedo_03       dc.b $00 ; 
TabMoveTorpedo_05       dc.b $00 ; 
TabMoveTorpedo_07       dc.b $00 ; 
TabMoveTranspLeRi       equ  *   ; 
TabMoveTorpedo_08       dc.b $00 ; 
TabMoveTorpedo_09       dc.b $00 ; 
TabMoveTorpedo_0a       dc.b $00 ; 
TabMoveSpriteUpDo       dc.b $00 ; cave and planes
TabMoveTorpedo_02       dc.b $00 ; 
TabMoveTorpedo_04       dc.b $00 ; 
TabMoveTorpedo_06       dc.b $00 ; 

TabMoveTorpedo_Val_1    dc.b $00 ; 
TabMoveTorpedo_Val_2    dc.b $00 ; 
TabMoveTorpedo_Val_3    dc.b $00 ; 
TabMoveTorpedo_Val_4    dc.b $00 ; 

TabPlaneSpeedAppr       dc.b $00 ; 
TabPlaneSpeedApprNone     = $00  ;
TabPlaneSpeedApprSlow     = $02  ;
TabPlaneSpeedApprFast     = $03  ;
TabPlaneSpeed_Val_1     dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabSpriteStatus         dc.b $00 ; 
TabSpriteStatusFree       = $00  ; 
TabSpriteStatusUsed       = $01  ; 
TabSpriteStatusExpl       = $ff  ; 
TabP_ShotStatus         dc.b $00 ; 
TabFlagTankEneShoot     dc.b $00 ; 
TabFlagTankEneShootNo     = $00  ; 
TabFlagTankEneShootYes    = $01  ; 
TabBulletStatus         dc.b $00 ; enemy plane shot
TabFlagTankEneMove      dc.b $00 ; 
TabTranspStatus         dc.b $00 ; 
TabBulletStatus1        dc.b $00 ; player shot
TabBulletStatus2        dc.b $00 ; 

TabPlaneDistance        dc.b $00 ; 
TabPlaneDist_Ini          = $00  ; 
TabPlaneDist_Explode      = $ff  ; 
TabPlaneDist_Start        = $01  ; 
TabPlaneDist_Turn         = $02  ; 
TabPlaneDist_Approach1    = $03  ; 
TabPlaneDist_Approach2    = $04  ; 
TabPlaneDist_Approach3    = $05  ; 
TabPlaneDist_Leave        = $06  ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabSeaShipMsgDestroy    dc.b $00 ; 
TabSeaShipMsgDestroyNo    = $00  ; 
TabSeaShipMsgDestroyYes   = $01  ; 
TabWrkPlayerShot        dc.b $00 ; 

TabPlayerAmmo           dc.b $0a ; 

TabTargetHit            dc.b $00 ; 
TabTargetHitNo            = $00  ; 
TabTargetHitYes           = $01  ; 

TabBulletActive         dc.b $01 ; 
TabBulletActiveNo         = $00  ; 
TabBulletActiveYes        = $01  ; 
TabBulletDamage         dc.b $14 ; 
TabEnemyBulletPhase     dc.b $03 ; 
TabBulletEnemyDist      dc.b $a0 ; 
                        
TabSeaVesselActive      equ  *   ; 
TabSeaVesselActiveNo      = $00  ; 
TabSeaVesselActiveYes     = $01  ; 
TabSeaVesselActive_00   dc.b $01 ; not used - count starts at $01
TabSeaVesselActive_01   dc.b $01 ; 
TabSeaVesselActive_02   dc.b $01 ; 
TabSeaVesselActive_03   dc.b $01 ; 
TabSeaVesselActive_04   dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        dc.b $01 ; 
                        
TabFlagCarrierActive    dc.b $01 ; 
TabFlagCarrierActiveNo    = $00  ; 
TabFlagCarrierActiveYes   = $01  ; 

TabBulletWaterSplash    dc.b $00 ; 
TabBulletWaterSplashNo    = $00  ; 
TabBulletWaterSplashYes   = $01  ; 

TabBeachSpriteLevel     dc.b $05 ; 
TabBeachSpriteLevelTmp  dc.b $02 ; 
TabWaitTankEneShoot     dc.b $00 ; 
TabWaitTankEneShootNo     = $00  ; 
TabWaitTankEneShootYes    = $01  ; 
TabTankShotSprtNoSav    dc.b $00 ; 
TabTankShotSpeedEne     dc.b $03 ; 
TabTankScoreSpriteNo    dc.b $00 ; 

TabSavPlaneSpriteNo     equ  *   ; 
TabWrkValueSav          dc.b $07 ; 

TabWrkNoUse             dc.b $63 ; 
TabPlayerNoUse2         dc.b $a8 ; 

TabPlayersSaveArea      equ  *   ; 
TabPlayerLivesSav       dc.b $02 ; p1
                        dc.b $00 ; p2
                        
TabPlayerCavePassSav    dc.b $81 ; p1
                        dc.b $01 ; 
                        
TabPlayerTanksSav       dc.b $00 ; p1
                        dc.b $00 ; p2
                        
TabPlayerNoUseSav1      dc.b $00 ; .honz.unused
                        dc.b $00 ; 
                        
TabPlayerScoreLoSav     dc.b $20 ; .hbu001. - p1 - BASIC 820
                        dc.b $30 ; .hbu001. - p2 - BASIC 820
                        
TabPlayerShowScoresSav  dc.b $00 ; p1
                        dc.b $00 ; p2
                        
TabWrkMakeDecimal       dc.b $02 ; 
                        
TabPlayerHillTargsSav   dc.b $0a ; 
                        dc.b $0a ; 
                        
TabBeachSpriteLevelSav  dc.b $05 ; player 1
                        dc.b $05 ; player 2
                        
TabPlayerVesselActSav   dc.b $3e ; 
                        dc.b $3e ; 
                        
TabPlayerNoUseSav2      dc.b $a8 ; .honz.unused
                        dc.b $a8 ; 

TabPlayerShipsSav       dc.b $02 ; p1
                        dc.b $01 ; p2
                        
TabPlayerScoreHiSav     dc.b $01 ; .hbu001. - p1 - BASIC 820
                        dc.b $00 ; .hbu001. - p2 - BASIC 820
                        
                        dc.b $dd ; 
                        dc.b $60 ; 
                        
TabRndSav               equ  *   ; 

TabCaveShipDir          dc.b $55 ; 
TabCaveShipDirMin         = $00
TabCaveShipDirMax         = $04
TabCaveShipDirInit        = $01

TabTankShotSpeed        dc.b $04 ; 

TabTankAdaptPosX        equ  *   ; 
TabTankAdaptPosX_00     dc.b $dd ; 
TabTankAdaptPosX_01     dc.b $00 ; 
TabTankAdaptPosX_02     dc.b $00 ; 
TabTankAdaptPosX_03     dc.b $00 ; 
TabTankAdaptPosX_04     dc.b $00 ; 
TabTankAdaptPosX_05     dc.b $00 ; 
TabTankAdaptPosX_06     dc.b $00 ; 
TabTankAdaptPosX_07     dc.b $00 ; 
                        
TabPosEnemyCannon       dc.b $a2 ; possible posX depending on TabBeachSpriteLevel
                        dc.b $a2 ; 
                        dc.b $76 ; 
                        dc.b $a2 ; 
                        dc.b $4a ; 
                        dc.b $9a ; 
                        dc.b $14 ; 
                        dc.b $a2 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabPosEnemyTank         dc.b $c8 ; possible posX depending on TabBeachSpriteLevel
                        dc.b $9a ; 
                        dc.b $9a ; 
                        dc.b $a2 ; 
                        dc.b $6c ; 
                        dc.b $a2 ; 
                        dc.b $3c ; 
                        dc.b $a2 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabPosEnemyGunner       dc.b $b4 ; possible posX depending on TabBeachSpriteLevel
                        dc.b $92 ; 
                        dc.b $80 ; 
                        dc.b $95 ; 
                        dc.b $55 ; 
                        dc.b $97 ; 
                        dc.b $21 ; 
                        dc.b $a2 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabPosEnemyTurret       dc.b $bf ; possible posX depending on TabBeachSpriteLevel
                        dc.b $b2 ; 
                        dc.b $8e ; 
                        dc.b $ae ; 
                        dc.b $60 ; 
                        dc.b $82 ; 
                        dc.b $36 ; 
                        dc.b $84 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabSpriteActiveCave     equ  *   ;
TabSpriteActivePlane    equ  *   ; 
TabSpriteActiveTank     equ  *   ;
TabSpriteActiveNo         = $00  ;
TabSpriteActiveYes        = $01  ;
TabSpriteActiveAll      dc.b $00 ; 
TabSpriteActiveBulletP1 dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
TabPlaneVal_3           dc.b $00 ; 
TabTranspActive         dc.b $00 ; 
TabTranspActiveNo         = $00  ;
TabTranspActiveYes        = $01  ;
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
TabPlaneWaitHori        dc.b $0a ; 
                        dc.b $1f ; 
                        dc.b $3b ; 
                        dc.b $03 ; 
                        dc.b $06 ; 
                        dc.b $0d ; 
                        dc.b $2d ; 
                        dc.b $07 ; 
                        
TabPlaneWaitVert        dc.b $2c ; 
                        dc.b $39 ; 
                        dc.b $0f ; 
                        dc.b $0b ; 
                        dc.b $36 ; 
                        dc.b $14 ; 
                        dc.b $05 ; 
                        dc.b $2b ; 
                        
TabPlaneWaitAppr        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $ff ; 
                        dc.b $16 ; 
                        dc.b $32 ; 
                        dc.b $16 ; 
                        dc.b $21 ; 

TabMoveSpritesStep      dc.b $00 ; 
TabMoveSpritesStepTime    = $3c  ; 

TabPlaneFreeSprite      dc.b $05 ; 
TabPlaneTranspWait      dc.b $03 ; 

TabPlaneTranspStatus    dc.b $00 ; 
TabPlaneTranspStatusFree  = $00  ; 
TabPlaneTranspStatusUsed  = $01  ; 

TabPlaneTranspWasHit    dc.b $00 ; 
TabPlaneTranspWasHitNo    = $00  ; 
TabPlaneTranspWasHitYes   = $01  ; 
TabPlaneTranspWasHitEnd   = $ff  ; 

TabRndSeed              dc.b $aa ; 

TabPlaneTurn            dc.b $02 ; 
TabPlaneTurnUpLe          = $01  ; 
TabPlaneTurnUp            = $02  ; 
TabPlaneTurnUpRi          = $03  ; 
TabTargetShipHit        dc.b $ff ; 
TabTargetShipHitNo        = $ff  ; 
TabTargetShipHitCarr      = $05  ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        dc.b $00 ; 
                        
                        dc.b $00 ; 
                        dc.b $00 ; 
TabShipCannonDegree     dc.b $01 ; 
TabSfxToPlay            dc.b $20 ; 
TabVoc1FreqLo           dc.b $a0 ; 
TabVoc3FreqHi           dc.b $bd ; 
TabVoc2AttDec           dc.b $0c ; 
TabVoc2FreqHi           dc.b $08 ; 
TabFilterCutHi          dc.b $5a ; 
; -------------------------------------------------------------------------------------------------------------- ;
TabSetFilterCutHi       dc.b $00 ; 
                        dc.b $ff ; 
                        dc.b $80 ; 
                        dc.b $70 ; 
                        dc.b $60 ; 
                        dc.b $70 ; 
                        dc.b $70 ; 
TabSetFilterCutHi_5a    dc.b $5a ; 
; -------------------------------------------------------------------------------------------------------------- ;
GraphicsData            include asm\Graphic.asm  ; Screens and character sets
; -------------------------------------------------------------------------------------------------------------- ;
